<?php
declare(strict_types=1);

date_default_timezone_set("America/Regina");

// --- Enable errors temporarily for debugging ---
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// --- Connect to database ---
include 'connect.php';
if (!isset($conn) || !$conn) {
    die("Database connection failed.");
}

$today = date("Y-m-d");
$month = date("Y-m");

function pick($a, ...$keys) {
    foreach ($keys as $k) {
        if (isset($a[$k]) && $a[$k] !== '') return $a[$k];
    }
    return null;
}
function ts($d, $t) { return $t ? strtotime("$d $t") : null; }
function fmt12($d, $t) { return $t ? date("g:i A", strtotime("$d $t")) : '—'; }

// --- Fetch today and tomorrow rows safely ---
function fetchPrayerRow($conn, $date) {
    $stmt = $conn->prepare("SELECT * FROM prayer_times WHERE prayer_date=? LIMIT 1");
    $stmt->bind_param("s", $date);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    return $row ?: [];
}
$todayRow = fetchPrayerRow($conn, $today);
$tomorrow = date("Y-m-d", strtotime("+1 day"));
$tomRow = fetchPrayerRow($conn, $tomorrow);

// --- Today's times ---
$fajrB = pick($todayRow,'fajar_begins'); $fajrI = pick($todayRow,'fajar');
$sun   = pick($todayRow,'sunrise');
$zuhrB = pick($todayRow,'zuhar_begins'); $zuhrI = pick($todayRow,'zuhar');
$asrB  = pick($todayRow,'asar_begins');  $asrI = pick($todayRow,'asar');
$magB  = pick($todayRow,'magrib_begins'); $magI = $magB;
$ishaB = pick($todayRow,'isha_begins');   $ishaI = pick($todayRow,'isha');
$tomFajrB = pick($tomRow,'fajar_begins'); $tomFajrI = pick($tomRow,'fajar');

$now = time();

// --- Current Salah ---
$current = ['name'=>'—','begin'=>null,'iqama'=>null];
if ($fajrB && $sun && $now >= ts($today,$fajrB) && $now < ts($today,$sun)) {
    $current = ['name'=>'Fajr','begin'=>$fajrB,'iqama'=>$fajrI];
} elseif ($zuhrB && $asrB && $now >= ts($today,$zuhrB) && $now < ts($today,$asrB)) {
    $current = ['name'=>'Dhuhr','begin'=>$zuhrB,'iqama'=>$zuhrI];
} elseif ($asrB && $magB && $now >= ts($today,$asrB) && $now < ts($today,$magB)) {
    $current = ['name'=>'Asr','begin'=>$asrB,'iqama'=>$asrI];
} elseif ($magB && $ishaB && $now >= ts($today,$magB) && $now < ts($today,$ishaB)) {
    $current = ['name'=>'Maghrib','begin'=>$magB,'iqama'=>$magI];
} elseif ($ishaB && $tomFajrB && $now >= ts($today,$ishaB) && $now < ts($tomorrow,$tomFajrB)) {
    $current = ['name'=>'Isha','begin'=>$ishaB,'iqama'=>$ishaI];
}

// --- Next Salah ---
$candidates = [
    ['name'=>'Fajr', 'b'=>$fajrB,'i'=>$fajrI,'d'=>$today],
    ['name'=>'Dhuhr','b'=>$zuhrB,'i'=>$zuhrI,'d'=>$today],
    ['name'=>'Asr',  'b'=>$asrB, 'i'=>$asrI,'d'=>$today],
    ['name'=>'Maghrib','b'=>$magB,'i'=>$magI,'d'=>$today],
    ['name'=>'Isha','b'=>$ishaB,'i'=>$ishaI,'d'=>$today],
    ['name'=>'Fajr (Tomorrow)','b'=>$tomFajrB,'i'=>$tomFajrI,'d'=>$tomorrow]
];
$next = null;
foreach ($candidates as $c) {
    if (!$c['b']) continue;
    if ($now < ts($c['d'],$c['b'])) { $next=$c; break; }
}
if (!$next) $next = end($candidates);

// --- Countdown target ---
$currentIqamaTs = $current['iqama'] ? ts($today,$current['iqama']) : null;
if ($current['name'] !== '—' && $currentIqamaTs && $now < $currentIqamaTs) {
    $targetName = $current['name']." (Iqama)";
    $targetTs   = $currentIqamaTs;
} else {
    $targetName = $next['name']." (Iqama)";
    $targetTs   = ($next['i'] ? ts($next['d'],$next['i']) : ts($next['d'],$next['b'])) ?: $now+60;
}

// --- Daily rows ---
$dailyRows = [
    ['Fajr',    fmt12($today,$fajrB), fmt12($today,$fajrI)],
    ['Dhuhr',   fmt12($today,$zuhrB), fmt12($today,$zuhrI)],
    ['Asr',     fmt12($today,$asrB),  fmt12($today,$asrI)],
    ['Maghrib', fmt12($today,$magB),  fmt12($today,$magI)],
    ['Isha',    fmt12($today,$ishaB), fmt12($today,$ishaI)],
];

// --- Monthly rows ---
$monthStart = $month.'-01';
$monthEnd = date("Y-m-t", strtotime($monthStart));
$monthRows = [];
$stmt = $conn->prepare("SELECT * FROM prayer_times WHERE prayer_date BETWEEN ? AND ? ORDER BY prayer_date ASC");
$stmt->bind_param("ss",$monthStart,$monthEnd);
$stmt->execute();
$res = $stmt->get_result();
while($row = $res->fetch_assoc()) $monthRows[] = $row;
$stmt->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Salah Times - Regina</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<style>
body{background:#064635;color:#fff;text-align:center;padding:20px;font-family:sans-serif}
.card{background:rgba(255,255,255,.1);border-radius:15px;padding:20px;max-width:800px;margin:auto}
.countdown{font-size:2em;font-weight:bold;margin:10px 0}
.table{color:white}
.table thead{background:#04502c}
.table-striped>tbody>tr:nth-of-type(odd){background:rgba(255,255,255,.08)}
.highlight{background:rgba(255,215,0,.2)!important}
.btn-light{margin:5px;color:#064635;font-weight:bold}
</style>
</head>
<body>
<div class="card">
<h2 class="text-white">🕌 Salah Times (Regina, Canada)</h2>
<h4 class="text-white"><?php echo date("l, d M Y h:i A"); ?></h4>
<hr style="border-color:white">

<h3 class="text-white">Current Salah: <span class="text-warning"><?php echo $current['name']; ?></span></h3>
<p class="text-white">Begin: <b><?php echo fmt12($today,$current['begin']); ?></b> | Iqama: <b><?php echo fmt12($today,$current['iqama']); ?></b></p>

<h3 class="text-white">Next Salah: <span class="text-danger"><?php echo $next['name']; ?></span></h3>
<p class="text-white">Begin: <b><?php echo fmt12($next['d'],$next['b']); ?></b> | Iqama: <b><?php echo fmt12($next['d'],$next['i']); ?></b></p>

<h3 class="text-white">Countdown to: <span class="text-info"><?php echo $targetName; ?></span></h3>
<div id="countdown" class="countdown text-white">Loading…</div>

<div class="d-grid gap-2 d-sm-flex justify-content-sm-center mt-3">
<button class="btn btn-light" data-bs-toggle="modal" data-bs-target="#dailyModal">All Day Timings</button>
<button class="btn btn-light" data-bs-toggle="modal" data-bs-target="#monthModal">All Month Timings</button>
</div>
</div>

<!-- Daily Modal -->
<div class="modal fade" id="dailyModal">
<div class="modal-dialog modal-lg modal-dialog-centered">
<div class="modal-content bg-dark text-white">
<div class="modal-header border-0">
<h5 class="modal-title">Today's Salah Timings</h5>
<button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
</div>
<div class="modal-body">
<table class="table table-striped text-center">
<thead><tr><th>Salah</th><th>Begin</th><th>Iqama</th></tr></thead>
<tbody>
<?php foreach($dailyRows as $r): ?>
<tr class="<?php echo ($r[0]===$current['name'])?'highlight':''; ?>">
<td><?php echo $r[0]; ?></td>
<td><?php echo $r[1]; ?></td>
<td><?php echo $r[2]; ?></td>
</tr>
<?php endforeach; ?>
</tbody>
</table>
</div>
</div>
</div>
</div>

<!-- Month Modal -->
<div class="modal fade" id="monthModal">
<div class="modal-dialog modal-lg modal-dialog-scrollable">
<div class="modal-content bg-dark text-white">
<div class="modal-header border-0">
<h5 class="modal-title">This Month's Salah Timings</h5>
<button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
</div>
<div class="modal-body">
<table class="table table-striped text-center">
<thead>
<tr>
<th>Date</th>
<th>Fajr<br><small>(Begin / Iqama)</small></th>
<th>Dhuhr<br><small>(Begin / Iqama)</small></th>
<th>Asr<br><small>(Begin / Iqama)</small></th>
<th>Maghrib<br><small>(Begin=Iqama)</small></th>
<th>Isha<br><small>(Begin / Iqama)</small></th>
</tr>
</thead>
<tbody>
<?php foreach($monthRows as $r): ?>
<tr>
<td><?php echo $r['prayer_date']; ?></td>
<td><?php echo fmt12($r['prayer_date'],$r['fajar_begins']); ?> / <?php echo fmt12($r['prayer_date'],$r['fajar']); ?></td>
<td><?php echo fmt12($r['prayer_date'],$r['zuhar_begins']); ?> / <?php echo fmt12($r['prayer_date'],$r['zuhar']); ?></td>
<td><?php echo fmt12($r['prayer_date'],$r['asar_begins']); ?> / <?php echo fmt12($r['prayer_date'],$r['asar']); ?></td>
<td><?php echo fmt12($r['prayer_date'],$r['magrib_begins']); ?></td>
<td><?php echo fmt12($r['prayer_date'],$r['isha_begins']); ?> / <?php echo fmt12($r['prayer_date'],$r['isha']); ?></td>
</tr>
<?php endforeach; ?>
</tbody>
</table>
</div>
</div>
</div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
const target = <?php echo $targetTs*1000; ?>;
function tick(){
  const el = document.getElementById('countdown');
  const diff = target - Date.now();
  if(diff <= 0){ el.textContent = "It's time!"; return; }
  const h = Math.floor((diff % (1000*60*60*24)) / (1000*60*60));
  const m = Math.floor((diff % (1000*60*60)) / (1000*60));
  const s = Math.floor((diff % (1000*60)) / 1000);
  el.textContent = h+"h "+m+"m "+s+"s";
}
setInterval(tick,1000); tick();
</script>
</body>
</html>
