<?php
ob_start();
// salah_times.php
declare(strict_types=1);

session_start();

// ✅ SECURITY CHECK 1: Must be logged in
if (!isset($_SESSION['user_id'])) {
  header("Location: index.php");
  exit;
}

// ✅ SECURITY CHECK: Allow only admin OR newsletterController (C is capital)
if (!isset($_SESSION['role']) ||
    (strtolower($_SESSION['role']) !== 'admin')) {
  header("Location: index.php");
  exit;
}

require_once __DIR__ . '/db.php';

header('Content-Type: application/json; charset=utf-8');
// Basic CORS (adjust domains if you need to restrict)
header('Access-Control-Allow-Origin: *');
header('X-Content-Type-Options: nosniff');

// Simple rate-limit headers (informational only)
header('X-RateLimit-Limit: 1000');
header('X-RateLimit-Remaining: 1000');

// Enforce API key if defined in db.php
if (defined('API_KEY') && API_KEY !== '') {
    $clientKey = $_GET['key'] ?? '';
    if (!hash_equals(API_KEY, (string)$clientKey)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        exit;
    }
}

$pdo = get_db();

// helper: validate date string YYYY-MM-DD
function is_valid_date(string $d): bool {
    $dt = DateTime::createFromFormat('Y-m-d', $d);
    return $dt && $dt->format('Y-m-d') === $d;
}

// helper: validate month string YYYY-MM
function is_valid_month(string $m): bool {
    $dt = DateTime::createFromFormat('Y-m', $m);
    return $dt && $dt->format('Y-m') === $m;
}

// helper: send JSON + exit
function json_exit($data, int $httpCode = 200): void {
    http_response_code($httpCode);
    echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// Prevent extremely large queries: max rows for month/range
define('MAX_ROWS', 366); // max number of rows we return for month/range

try {
    // If month provided: return whole month
    if (isset($_GET['month'])) {
        $month = trim((string)$_GET['month']);
        if (!is_valid_month($month)) {
            json_exit(['error' => 'Invalid month format. Use YYYY-MM'], 400);
        }

        // compute start and end date of the month
        $start = (new DateTime($month . '-01'))->format('Y-m-d');
        $end = (new DateTime($start))->modify('last day of this month')->format('Y-m-d');

        // enforce reasonable range (avoid open-ended)
        $interval = (new DateTime($start))->diff(new DateTime($end))->days + 1;
        if ($interval > MAX_ROWS) {
            json_exit(['error' => 'Requested month range too large.'], 400);
        }

        $sql = "SELECT prayer_date, fajar_begins, fajar, sunrise, zuhar_begins, zuhar,
                       asar_begins, asar, sunset, magrib_begins, isha_begins, isha,
                       jumah1, jumah2, jumah3, jumah4
                FROM prayer_times
                WHERE prayer_date BETWEEN :start AND :end
                ORDER BY prayer_date ASC
                LIMIT :limit";

        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':start', $start, PDO::PARAM_STR);
        $stmt->bindValue(':end', $end, PDO::PARAM_STR);
        $stmt->bindValue(':limit', (int)MAX_ROWS, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll();

        if (count($rows) === 0) {
            json_exit(['message' => "No data found for month $month"], 404);
        }

        json_exit($rows);
    }

    // If start & end provided (range)
    if (isset($_GET['start']) && isset($_GET['end'])) {
        $start = trim((string)$_GET['start']);
        $end = trim((string)$_GET['end']);
        if (!is_valid_date($start) || !is_valid_date($end)) {
            json_exit(['error' => 'Invalid date format. Use YYYY-MM-DD for start and end'], 400);
        }
        $startDt = new DateTime($start);
        $endDt = new DateTime($end);
        if ($startDt > $endDt) {
            json_exit(['error' => 'start date must be <= end date'], 400);
        }
        $days = $startDt->diff($endDt)->days + 1;
        if ($days > MAX_ROWS) {
            json_exit(['error' => 'Requested range too large. Limit ' . MAX_ROWS . ' days.'], 400);
        }

        $sql = "SELECT prayer_date, fajar_begins, fajar, sunrise, zuhar_begins, zuhar,
                       asar_begins, asar, sunset, magrib_begins, isha_begins, isha,
                       jumah1, jumah2, jumah3, jumah4
                FROM prayer_times
                WHERE prayer_date BETWEEN :start AND :end
                ORDER BY prayer_date ASC";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':start' => $start, ':end' => $end]);
        $rows = $stmt->fetchAll();
        if (count($rows) === 0) {
            json_exit(['message' => "No data found between $start and $end"], 404);
        }
        json_exit($rows);
    }

    // Single date fetch (default to today)
    $date = isset($_GET['date']) ? trim((string)$_GET['date']) : (new DateTime())->format('Y-m-d');
    if (!is_valid_date($date)) {
        json_exit(['error' => 'Invalid date format. Use YYYY-MM-DD'], 400);
    }

    // Optional: disallow requests outside a reasonable window (e.g., 1970-01-01..2100-12-31)
    $min = '1970-01-01';
    $max = '2100-12-31';
    if ($date < $min || $date > $max) {
        json_exit(['error' => 'Date out of allowed range'], 400);
    }

    $sql = "SELECT prayer_date, fajar_begins, fajar, sunrise, zuhar_begins, zuhar,
                   asar_begins, asar, sunset, magrib_begins, isha_begins, isha,
                   jumah1, jumah2, jumah3, jumah4
            FROM prayer_times
            WHERE prayer_date = :date
            LIMIT 1";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([':date' => $date]);
    $row = $stmt->fetch();

    if (!$row) {
        json_exit(['message' => "No data found for date $date"], 404);
    }

    json_exit($row);

} catch (Throwable $e) {
    // Log the error server-side if you have logging (not printed to client)
    // error_log($e->getMessage());

    json_exit(['error' => 'Internal server error'], 500);
}
