<?php
ob_start();
include "layout.php";
require_once "db.php";

session_start();

// ✅ SECURITY CHECK 1: Must be logged in
if (!isset($_SESSION['user_id'])) {
  header("Location: index.php");
  exit;
}

// ✅ SECURITY CHECK: Allow only admin OR newsletterController (C is capital)
if (!isset($_SESSION['role']) ||
    (strtolower($_SESSION['role']) !== 'admin')) {
  header("Location: index.php");
  exit;
}

$pdo = get_db();

// ✅ Handle Delete
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id'])) {
    $id = (int)$_POST['delete_id'];
    $pdo->prepare("DELETE FROM education_students WHERE id=?")->execute([$id]);
    echo "<div class='alert alert-danger text-center mt-3'>🗑️ Student deleted successfully!</div>";
}

// ✅ Handle Edit Save
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_id'])) {
    $stmt = $pdo->prepare("UPDATE education_students 
        SET student_name=?, parent_name=?, parent_email=?, parent_phone=?, amount_paid=?, payment_method=?, entry_date=?, expiry_date=? 
        WHERE id=?");
    $stmt->execute([
        $_POST['student_name'],
        $_POST['parent_name'],
        $_POST['parent_email'],
        $_POST['parent_phone'],
        $_POST['amount_paid'],
        $_POST['payment_method'],
        $_POST['entry_date'],
        $_POST['expiry_date'],
        $_POST['update_id']
    ]);
    echo "<div class='alert alert-success text-center mt-3'>✅ Student updated successfully!</div>";
}

// ✅ Handle Renewal
if (isset($_GET['renew'])) {
    $id = (int)$_GET['renew'];
    $row = $pdo->query("SELECT frequency, expiry_date FROM education_students WHERE id=$id")->fetch();
    if ($row) {
        $expiry = new DateTime($row['expiry_date']);
        if ($row['frequency'] === 'weekly') $expiry->modify('+7 days');
        else $expiry->modify('+1 month');
        $pdo->prepare("UPDATE education_students SET expiry_date=? WHERE id=?")->execute([$expiry->format('Y-m-d'), $id]);
        echo "<div class='alert alert-success text-center mt-3'>🔁 Student program renewed successfully!</div>";
    }
}

// ✅ Fetch all programs
$programs = $pdo->query("SELECT * FROM education_programs ORDER BY program_name ASC")->fetchAll();

// ✅ Handle New Registration
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['update_id']) && !isset($_POST['delete_id'])) {
    $students = $_POST['student_name'];
    $program_id = $_POST['program_id'];
    $parent_name = trim($_POST['parent_name']);
    $parent_email = trim($_POST['parent_email']);
    $parent_phone = trim($_POST['parent_phone']);
    $payment_method = $_POST['payment_method'];
    $entry_date = $_POST['entry_date'];
    $amount_paid = (float)$_POST['amount_paid'];

    $stmt = $pdo->prepare("SELECT program_name, amount_monthly, frequency FROM education_programs WHERE id=?");
    $stmt->execute([$program_id]);
    $program = $stmt->fetch();

    $expiry_date = date('Y-m-d', strtotime($entry_date . ($program['frequency'] === 'weekly' ? ' +7 days' : ' +1 month')));

    foreach ($students as $student_name) {
        if (trim($student_name) === '') continue;
        $insert = $pdo->prepare("INSERT INTO education_students 
            (student_name, parent_name, parent_email, parent_phone, program_id, program_name, price, frequency, amount_paid, payment_method, entry_date, expiry_date)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $insert->execute([
            $student_name,
            $parent_name,
            $parent_email,
            $parent_phone,
            $program_id,
            $program['program_name'],
            $program['amount_monthly'],
            $program['frequency'],
            $amount_paid,
            $payment_method,
            $entry_date,
            $expiry_date
        ]);
    }
    echo "<div class='alert alert-success text-center mt-3'>✅ Student(s) registered successfully!</div>";
}

// ✅ Fetch all registered students
$students = $pdo->query("SELECT * FROM education_students ORDER BY created_at DESC")->fetchAll();
?>

<div class="container-fluid mt-5">
  <h3><i class="bi bi-people-fill"></i> Education Controller – Student Registrations</h3>

  <!-- Registration Form -->
  <div class="card mt-3 p-4 shadow-sm">
    <h5 class="text-white"><i class="bi bi-person-plus-fill text-success"></i> Register New Student(s)</h5>
    <form method="post" class="row g-3">
      <div class="col-md-4">
        <label class="form-label text-white">Parent Name</label>
        <input type="text" name="parent_name" class="form-control" required>
      </div>
      <div class="col-md-4">
        <label class="form-label text-white">Parent Email</label>
        <input type="email" name="parent_email" class="form-control">
      </div>
      <div class="col-md-4">
        <label class="form-label text-white">Parent Phone</label>
        <input type="text" name="parent_phone" class="form-control" required>
      </div>

      <div class="col-md-6">
        <label class="form-label text-white">Select Program</label>
        <select name="program_id" id="programSelect" class="form-select" required onchange="updateProgramInfo()">
          <option value="">-- Select Program --</option>
          <?php foreach ($programs as $p): ?>
            <option value="<?= $p['id'] ?>" data-price="<?= $p['amount_monthly'] ?>" data-frequency="<?= $p['frequency'] ?>">
              <?= htmlspecialchars($p['program_name']) ?> (<?= ucfirst($p['frequency']) ?>)
            </option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label text-white">Program Price (CAD)</label>
        <input type="text" id="programPrice" class="form-control" readonly>
      </div>

      <div class="col-md-3">
        <label class="form-label text-white">Amount Paid (CAD)</label>
        <input type="number" step="0.01" name="amount_paid" id="amountPaid" class="form-control" required>
      </div>

      <div class="col-md-3">
        <label class="form-labe text-whitel">Entry Date</label>
        <input type="date" name="entry_date" id="entryDate" class="form-control" required onchange="autoSetExpiry()">
      </div>

      <div class="col-md-3">
        <label class="form-label text-white">Expiry Date</label>
        <input type="date" name="expiry_date" id="expiryDate" class="form-control" readonly>
      </div>

      <div class="col-md-3">
        <label class="form-label text-white">Payment Method</label>
        <select name="payment_method" class="form-select" required>
          <option value="cash">Cash</option>
          <option value="card">Card</option>
          <option value="online">Online</option>
        </select>
      </div>

      <div class="col-md-12">
        <label class="form-label text-white">Student Name(s)</label>
        <div id="studentFields">
          <input type="text" name="student_name[]" class="form-control mb-2" placeholder="Enter student name" required>
        </div>
        <button type="button" class="btn btn-outline-primary btn-sm" onclick="addStudentField()"><i class="bi bi-plus-circle"></i> Add More Student</button>
      </div>

      <div class="col-md-12 text-end mt-3">
        <button class="btn btn-success"><i class="bi bi-check-circle"></i> Register</button>
      </div>
    </form>
  </div>

  <!-- Student List -->
  <div class="card mt-4 p-3 shadow-sm">
    <h5 class="text-white"><i class="bi bi-table"></i> Registered Students</h5>
    <div class="table-responsive">
      <table class="table table-striped table-bordered align-middle text-center">
        <thead class="table-dark">
          <tr>
            <th>#</th><th>Student</th><th>Program</th><th>Parent</th><th>Payment</th>
            <th>Join</th><th>Expiry</th><th>Type</th><th>Price</th><th>Paid</th><th>Actions</th>
          </tr>
        </thead>
        <tbody>
          <?php if ($students): foreach ($students as $i => $s): 
            $today = new DateTime();
            $expiry = new DateTime($s['expiry_date']);
            $diff = $today->diff($expiry)->days;
            $expired = $expiry < $today;
            $statusBadge = $expired 
              ? "<span class='badge bg-danger'>Expired</span>"
              : ($diff <= 3 ? "<span class='badge bg-warning text-dark'>Expiring Soon</span>" 
                            : "<span class='badge bg-success'>Active</span>");
          ?>
            <tr id="student-<?= $s['id'] ?>">
              <td><?= $i + 1 ?></td>
              <td><?= htmlspecialchars($s['student_name']) ?></td>
              <td><?= htmlspecialchars($s['program_name']) ?></td>
              <td><?= htmlspecialchars($s['parent_name']) ?><br><small><?= htmlspecialchars($s['parent_phone']) ?></small></td>
              <td><span class="badge bg-<?= $s['payment_method']=='cash'?'secondary':($s['payment_method']=='card'?'info':'success') ?>"><?= ucfirst($s['payment_method']) ?></span></td>
              <td><?= htmlspecialchars($s['entry_date']) ?></td>
              <td><?= htmlspecialchars($s['expiry_date']) ?><br><?= $statusBadge ?></td>
              <td><span class="badge bg-warning text-dark"><?= ucfirst($s['frequency']) ?></span></td>
              <td>CAD <?= number_format($s['price'], 2) ?></td>
              <td>CAD <?= number_format($s['amount_paid'], 2) ?></td>
              <td>
                <button type="button" class="btn btn-sm btn-primary" 
                  onclick="openEditModal(<?= $s['id'] ?>,'<?= addslashes($s['student_name']) ?>','<?= addslashes($s['parent_name']) ?>','<?= addslashes($s['parent_email']) ?>','<?= addslashes($s['parent_phone']) ?>','<?= $s['amount_paid'] ?>','<?= $s['payment_method'] ?>','<?= $s['entry_date'] ?>','<?= $s['expiry_date'] ?>')">
                  <i class="bi bi-pencil-square"></i>
                </button>
                <button type="button" class="btn btn-sm btn-danger" onclick="openDeleteModal(<?= $s['id'] ?>)">
                  <i class="bi bi-trash"></i>
                </button>
                <a href="?renew=<?= $s['id'] ?>" class="btn btn-sm btn-success"><i class="bi bi-arrow-repeat"></i></a>
              </td>
            </tr>
          <?php endforeach; else: ?>
            <tr><td colspan="11">No students registered yet.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<!-- ✏️ Edit Modal -->
<div class="modal fade" id="editModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <form method="post">
        <div class="modal-header bg-success text-white">
          <h5 class="modal-title"><i class="bi bi-pencil-square"></i> Edit Student</h5>
          <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <input type="hidden" name="update_id" id="editId">
          <div class="mb-2 text-dark"><label>Student Name</label><input type="text" name="student_name" id="editStudent" class="form-control" required></div>
          <div class="mb-2 text-dark"><label>Parent Name</label><input type="text" name="parent_name" id="editParent" class="form-control" required></div>
          <div class="mb-2 text-dark"><label>Parent Email</label><input type="email" name="parent_email" id="editEmail" class="form-control"></div>
          <div class="mb-2 text-dark"><label>Parent Phone</label><input type="text" name="parent_phone" id="editPhone" class="form-control" required></div>
          <div class="mb-2 text-dark"><label>Amount Paid (CAD)</label><input type="number" step="0.01" name="amount_paid" id="editPaid" class="form-control" required></div>
          <div class="mb-2 text-dark"><label>Payment Method</label>
            <select name="payment_method" id="editMethod" class="form-select">
              <option value="cash">Cash</option><option value="card">Card</option><option value="online">Online</option>
            </select></div>
          <div class="mb-2 text-dark"><label>Entry Date</label><input type="date" name="entry_date" id="editEntry" class="form-control" required></div>
          <div class="mb-2 text-dark"><label>Expiry Date</label><input type="date" name="expiry_date" id="editExpiry" class="form-control" required></div>
        </div>
        <div class="modal-footer">
          <button class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
          <button class="btn btn-success"><i class="bi bi-save"></i> Save</button>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- 🗑️ Delete Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <form method="post">
        <input type="hidden" name="delete_id" id="deleteId">
        <div class="modal-header bg-danger text-white">
          <h5 class="modal-title"><i class="bi bi-trash"></i> Confirm Delete</h5>
          <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body text-center text-dark">
          <p>Are you sure you want to delete this student record?</p>
        </div>
        <div class="modal-footer">
          <button class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
          <button class="btn btn-danger"><i class="bi bi-trash"></i> Delete</button>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- ✅ Correct Bootstrap Bundle -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>

<script>
function updateProgramInfo() {
  const select = document.getElementById('programSelect');
  const price = select.options[select.selectedIndex]?.getAttribute('data-price');
  const freq = select.options[select.selectedIndex]?.getAttribute('data-frequency');
  document.getElementById('programPrice').value = price ? parseFloat(price).toFixed(2) + ' (' + freq + ')' : '';
  autoSetExpiry();
  updateTotalAmount(); // recalc total
}

function autoSetExpiry() {
  const entry = document.getElementById('entryDate').value;
  const select = document.getElementById('programSelect');
  const freq = select.options[select.selectedIndex]?.getAttribute('data-frequency');
  if (!entry || !freq) return;
  const d = new Date(entry);
  if (freq === 'weekly') d.setDate(d.getDate() + 7); else d.setMonth(d.getMonth() + 1);
  document.getElementById('expiryDate').value = d.toISOString().slice(0,10);
}

function addStudentField() {
  const div = document.createElement('div');
  div.innerHTML = '<input type="text" name="student_name[]" class="form-control mb-2 text-dark" placeholder="Enter student name" required>';
  document.getElementById('studentFields').appendChild(div);
  updateTotalAmount(); // recalc when new student added
}

// 🔹 Auto multiply logic
function updateTotalAmount() {
  const count = document.querySelectorAll('#studentFields input[name="student_name[]"]').length;
  const programSelect = document.getElementById('programSelect');
  const price = parseFloat(programSelect.options[programSelect.selectedIndex]?.getAttribute('data-price') || 0);
  const total = count * price;
  const amountField = document.getElementById('amountPaid');
  if (!isNaN(total) && total > 0) amountField.value = total.toFixed(2);
}

// 🔹 Recalculate when changing program or editing student names
document.getElementById('studentFields').addEventListener('input', updateTotalAmount);

function openEditModal(id, student, parent, email, phone, paid, method, entry, expiry) {
  document.getElementById('editId').value = id;
  document.getElementById('editStudent').value = student;
  document.getElementById('editParent').value = parent;
  document.getElementById('editEmail').value = email;
  document.getElementById('editPhone').value = phone;
  document.getElementById('editPaid').value = paid;
  document.getElementById('editMethod').value = method;
  document.getElementById('editEntry').value = entry;
  document.getElementById('editExpiry').value = expiry;
  const modal = new bootstrap.Modal(document.getElementById('editModal'));
  modal.show();
}

function openDeleteModal(id) {
  document.getElementById('deleteId').value = id;
  const modal = new bootstrap.Modal(document.getElementById('deleteModal'));
  modal.show();
}
</script>
