<?php
ob_start();
require_once "db.php"; // must provide $pdo
session_start();

// ✅ SECURITY CHECK 1: Must be logged in
if (!isset($_SESSION['user_id'])) {
  header("Location: index.php");
  exit;
}

// ✅ SECURITY CHECK 2: Must be admin
if (!isset($_SESSION['role']) || strtolower($_SESSION['role']) !== 'admin') {
  header("Location: index.php");
  exit;
}

// ✅ Filters
$status = isset($_GET['status']) ? trim($_GET['status']) : '';
$q      = isset($_GET['q']) ? trim($_GET['q']) : '';

$where = [];
$params = [];

if ($status !== '' && in_array($status, ['pending','approved','rejected','contacted'], true)) {
  $where[] = "status = :status";
  $params[':status'] = $status;
}

if ($q !== '') {
  $where[] = "(business_name LIKE :q OR contact_person LIKE :q OR email LIKE :q OR phone LIKE :q OR business_category LIKE :q)";
  $params[':q'] = "%{$q}%";
}

$sql = "SELECT * FROM business_registrations";
if ($where) $sql .= " WHERE " . implode(" AND ", $where);
$sql .= " ORDER BY created_at DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

// ✅ Counts
$countStmt = $pdo->query("SELECT status, COUNT(*) c FROM business_registrations GROUP BY status");
$counts = ['pending'=>0,'approved'=>0,'rejected'=>0,'contacted'=>0];
foreach ($countStmt->fetchAll(PDO::FETCH_ASSOC) as $r) {
  $counts[$r['status']] = (int)$r['c'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Business Registrations | IAOS Admin</title>

<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">

<style>
body { background:#f5f6fa; }
.admin-wrapper { display:flex; min-height:100vh; }
.admin-sidebar { width:250px; background:#212529; color:#fff; }
.admin-sidebar a {
  color:#ccc; padding:12px 20px; display:block;
  text-decoration:none; border-bottom:1px solid #2f343a;
}
.admin-sidebar a:hover, .admin-sidebar a.active {
  background:#0d6efd; color:#fff;
}
.admin-content { flex-grow:1; padding:30px; }
.section-card { border-radius:10px; }
.section-title { font-weight:700; font-size:1.3rem; margin-bottom:15px; }

.badge-pending   { background:#ffc107; color:#000; }
.badge-approved  { background:#198754; }
.badge-rejected  { background:#dc3545; }
.badge-contacted { background:#0dcaf0; color:#000; }

.table td { vertical-align: middle; }
.small-link { font-size: 13px; }
</style>
</head>

<body>
<div class="admin-wrapper">

  <?php include "sidebar.php"; ?>

  <div class="admin-content">

    <div class="d-flex align-items-center justify-content-between mb-3">
      <h3 class="mb-0">Business Registrations</h3>
    </div>

    <!-- ✅ SUCCESS / ERROR MESSAGE -->
    <?php if(isset($_GET['saved'])): ?>
      <div class="alert alert-success">✅ Saved successfully!</div>
    <?php endif; ?>
    <?php if(isset($_GET['deleted'])): ?>
      <div class="alert alert-warning">🗑️ Deleted successfully!</div>
    <?php endif; ?>
    <?php if(isset($_GET['err'])): ?>
      <div class="alert alert-danger">❌ Something went wrong. Please try again.</div>
    <?php endif; ?>

    <!-- ✅ Small Stats -->
    <div class="row g-2 mb-3">
      <div class="col-md-3"><div class="p-3 bg-white rounded shadow-sm">Pending: <b><?= (int)$counts['pending'] ?></b></div></div>
      <div class="col-md-3"><div class="p-3 bg-white rounded shadow-sm">Approved: <b><?= (int)$counts['approved'] ?></b></div></div>
      <div class="col-md-3"><div class="p-3 bg-white rounded shadow-sm">Rejected: <b><?= (int)$counts['rejected'] ?></b></div></div>
      <div class="col-md-3"><div class="p-3 bg-white rounded shadow-sm">Contacted: <b><?= (int)$counts['contacted'] ?></b></div></div>
    </div>

    <!-- ✅ Filters -->
    <div class="card p-3 shadow-sm mb-3 section-card">
      <form class="row g-2 align-items-center" method="get" autocomplete="off">
        <div class="col-md-3">
          <select name="status" class="form-select">
            <option value="">All Status</option>
            <option value="pending"   <?= $status==='pending'?'selected':'' ?>>Pending</option>
            <option value="approved"  <?= $status==='approved'?'selected':'' ?>>Approved</option>
            <option value="rejected"  <?= $status==='rejected'?'selected':'' ?>>Rejected</option>
            <option value="contacted" <?= $status==='contacted'?'selected':'' ?>>Contacted</option>
          </select>
        </div>
        <div class="col-md-6">
          <input type="text" class="form-control" name="q" value="<?= htmlspecialchars($q) ?>" placeholder="Search business / person / email / phone / category">
        </div>
        <div class="col-md-3 d-flex gap-2">
          <button class="btn btn-primary w-50"><i class="bi bi-search"></i> Search</button>
          <a href="admin_business_registration.php" class="btn btn-outline-secondary w-50">Reset</a>
        </div>
      </form>
    </div>

    <!-- ✅ TABLE STARTS DIRECTLY -->
    <div class="card p-3 shadow-sm section-card">
      <div class="section-title mb-2">📋 Registrations</div>

      <div class="table-responsive">
        <table class="table table-hover align-middle">
          <thead class="table-light">
            <tr>
              <th>#</th>
              <th>Business</th>
              <th>Contact</th>
              <th>Category</th>
              <th>Offer</th>
              <th>Status</th>
              <th>Submitted</th>
              <th style="width:420px;">Actions</th>
            </tr>
          </thead>

          <tbody>
          <?php if(empty($rows)): ?>
            <tr>
              <td colspan="8" class="text-center text-muted py-4">No registrations found.</td>
            </tr>
          <?php endif; ?>

          <?php foreach($rows as $r): ?>
            <?php
              $badgeClass = 'badge-pending';
              if ($r['status'] === 'approved') $badgeClass = 'badge-approved';
              if ($r['status'] === 'rejected') $badgeClass = 'badge-rejected';
              if ($r['status'] === 'contacted') $badgeClass = 'badge-contacted';
            ?>
            <tr>
              <td><?= (int)$r['id'] ?></td>

              <td>
                <div class="fw-semibold"><?= htmlspecialchars($r['business_name']) ?></div>
                <div class="text-muted small"><?= htmlspecialchars($r['address']) ?></div>

                <?php if(!empty($r['website'])): ?>
                  <a class="small-link" href="<?= htmlspecialchars($r['website']) ?>" target="_blank">
                    <i class="bi bi-link-45deg"></i> Website
                  </a>
                <?php endif; ?>
              </td>

              <td>
                <div class="fw-semibold"><?= htmlspecialchars($r['contact_person']) ?></div>
                <div class="small">
                  <a href="mailto:<?= htmlspecialchars($r['email']) ?>"><i class="bi bi-envelope"></i> <?= htmlspecialchars($r['email']) ?></a><br>
                  <a href="tel:<?= htmlspecialchars($r['phone']) ?>"><i class="bi bi-telephone"></i> <?= htmlspecialchars($r['phone']) ?></a>
                </div>
              </td>

              <td><?= htmlspecialchars($r['business_category']) ?></td>

              <td>
                <div><?= htmlspecialchars($r['discount_offer']) ?></div>
                <?php if(!empty($r['special_conditions'])): ?>
                  <div class="text-muted small">Conditions: <?= htmlspecialchars($r['special_conditions']) ?></div>
                <?php endif; ?>
              </td>

              <td>
                <span class="badge <?= $badgeClass ?> text-uppercase"><?= htmlspecialchars($r['status']) ?></span>
              </td>

              <td class="small"><?= htmlspecialchars($r['created_at']) ?></td>

              <td>
                <div class="d-flex flex-wrap gap-2">

                  <!-- ✅ Approve -->
                  <form action="save_business_reg.php" method="POST">
                    <input type="hidden" name="action" value="approve">
                    <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                    <button class="btn btn-success btn-sm" type="submit">
                      <i class="bi bi-check2-circle"></i> Approve
                    </button>
                  </form>

                  <!-- ✅ Reject -->
                  <form action="save_business_reg.php" method="POST" onsubmit="return confirm('Reject this business?');">
                    <input type="hidden" name="action" value="reject">
                    <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                    <button class="btn btn-danger btn-sm" type="submit">
                      <i class="bi bi-x-circle"></i> Reject
                    </button>
                  </form>

                  <!-- ✅ Contacted -->
                  <form action="save_business_reg.php" method="POST">
                    <input type="hidden" name="action" value="contacted">
                    <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                    <button class="btn btn-info btn-sm" type="submit">
                      <i class="bi bi-telephone-outbound"></i> Contacted
                    </button>
                  </form>

                  <!-- ✅ Delete -->
                  <form action="save_business_reg.php" method="POST" onsubmit="return confirm('Delete this registration permanently?');">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                    <button class="btn btn-dark btn-sm" type="submit">
                      <i class="bi bi-trash"></i> Delete
                    </button>
                  </form>

                  <!-- ✅ Notes -->
                  <button class="btn btn-outline-secondary btn-sm"
                          data-bs-toggle="modal"
                          data-bs-target="#notesModal<?= (int)$r['id'] ?>">
                    <i class="bi bi-journal-text"></i> Notes
                  </button>

                </div>

                <!-- ✅ Notes Modal -->
                <div class="modal fade" id="notesModal<?= (int)$r['id'] ?>" tabindex="-1" aria-hidden="true">
                  <div class="modal-dialog modal-dialog-centered modal-lg">
                    <div class="modal-content">
                      <form action="save_business_reg.php" method="POST">
                        <div class="modal-header">
                          <h5 class="modal-title">Admin Notes - ID <?= (int)$r['id'] ?></h5>
                          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                          <input type="hidden" name="action" value="notes">
                          <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">

                          <div class="mb-2">
                            <label class="fw-bold">Notes</label>
                            <textarea name="admin_notes" class="form-control" rows="6"><?= htmlspecialchars($r['admin_notes'] ?? '') ?></textarea>
                          </div>
                        </div>
                        <div class="modal-footer">
                          <button class="btn btn-secondary" type="button" data-bs-dismiss="modal">Close</button>
                          <button class="btn btn-primary" type="submit">Save Notes</button>
                        </div>
                      </form>
                    </div>
                  </div>
                </div>

              </td>
            </tr>
          <?php endforeach; ?>
          </tbody>

        </table>
      </div>

    </div>

  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
