<?php

require_once "../../db.php"; // must provide get_db()

session_start();


// ✅ SECURITY CHECK 1: Must be logged in
if (!isset($_SESSION['user_id'])) {
  header("Location: index.php");
  exit;
}

// ✅ SECURITY CHECK 2: Must be admin
if (!isset($_SESSION['role']) || strtolower($_SESSION['role']) !== 'admin') {
  header("Location: index.php");
  exit;
}



$pdo = get_db();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

// Filters
$eventId  = isset($_GET['event_id']) ? (int)$_GET['event_id'] : 0;
$status   = trim($_GET['status'] ?? '');
$q        = trim($_GET['q'] ?? '');
$page     = max(1, (int)($_GET['page'] ?? 1));
$perPage  = 25;
$offset   = ($page - 1) * $perPage;

// Load events dropdown
$events = $pdo->query("SELECT id, title FROM events ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);

// Build WHERE
$where = " WHERE 1=1 ";
$params = [];

if ($eventId > 0) {
  $where .= " AND b.event_id = ? ";
  $params[] = $eventId;
}

if ($status !== '') {
  $where .= " AND b.booking_status = ? ";
  $params[] = $status;
}

if ($q !== '') {
  $where .= " AND (
    b.order_id LIKE ?
    OR b.customer_email LIKE ?
    OR b.paypal_email LIKE ?
    OR b.payer_name LIKE ?
    OR e.title LIKE ?
  ) ";
  $like = "%".$q."%";
  array_push($params, $like, $like, $like, $like, $like);
}

// Count total
$countStmt = $pdo->prepare("
  SELECT COUNT(*)
  FROM event_bookings b
  INNER JOIN events e ON e.id = b.event_id
  $where
");
$countStmt->execute($params);
$totalRows = (int)$countStmt->fetchColumn();
$totalPages = max(1, (int)ceil($totalRows / $perPage));

// Fetch data
$dataStmt = $pdo->prepare("
  SELECT
    b.*,
    e.title AS event_title,
    e.event_date
  FROM event_bookings b
  INNER JOIN events e ON e.id = b.event_id
  $where
  ORDER BY b.id DESC
  LIMIT $perPage OFFSET $offset
");
$dataStmt->execute($params);
$rows = $dataStmt->fetchAll(PDO::FETCH_ASSOC);

// Helpers
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function badge($status){
  $s = strtoupper(trim((string)$status));
  if ($s === 'CONFIRMED') return '<span class="badge bg-success">CONFIRMED</span>';
  if ($s === 'CANCELLED') return '<span class="badge bg-secondary">CANCELLED</span>';
  if ($s === 'REFUNDED')  return '<span class="badge bg-warning text-dark">REFUNDED</span>';
  return '<span class="badge bg-dark">'.h($s).'</span>';
}

function paypalBadge($status){
  $s = strtoupper(trim((string)$status));
  if ($s === 'COMPLETED') return '<span class="badge bg-success">COMPLETED</span>';
  return '<span class="badge bg-danger">'.h($s).'</span>';
}

// totals box (current filter)
$sumStmt = $pdo->prepare("
  SELECT
    COALESCE(SUM(b.ticket_count),0) AS total_tickets,
    COALESCE(SUM(b.amount),0) AS total_amount
  FROM event_bookings b
  INNER JOIN events e ON e.id=b.event_id
  $where
");
$sumStmt->execute($params);
$totals = $sumStmt->fetch(PDO::FETCH_ASSOC);
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Event Bookings</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body{background:#f4f6f8;}
    .card{border:1px solid #e5e7eb;border-radius:12px;}
    .table td, .table th{vertical-align:middle;}
    .mono{font-family:ui-monospace,SFMono-Regular,Menlo,Monaco,Consolas,monospace;font-size:12px;}
  </style>
</head>
<body>

<div class="container-xl py-4">

  <div class="d-flex justify-content-between align-items-center mb-3">
    <h3 class="mb-0">Event Bookings</h3>
    <a class="btn btn-outline-dark btn-sm" href="admin_about.php">Back</a>
  </div>

  <div class="card p-3 mb-3">
    <form class="row g-2 align-items-end" method="get">

      <div class="col-md-4">
        <label class="form-label fw-bold">Event</label>
        <select class="form-select" name="event_id">
          <option value="0">All Events</option>
          <?php foreach($events as $ev): ?>
            <option value="<?= (int)$ev['id'] ?>" <?= $eventId==(int)$ev['id']?'selected':'' ?>>
              #<?= (int)$ev['id'] ?> - <?= h($ev['title']) ?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label fw-bold">Booking Status</label>
        <select class="form-select" name="status">
          <option value="" <?= $status===''?'selected':'' ?>>All</option>
          <option value="CONFIRMED" <?= $status==='CONFIRMED'?'selected':'' ?>>CONFIRMED</option>
          <option value="REFUNDED" <?= $status==='REFUNDED'?'selected':'' ?>>REFUNDED</option>
          <option value="CANCELLED" <?= $status==='CANCELLED'?'selected':'' ?>>CANCELLED</option>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label fw-bold">Search</label>
        <input class="form-control" name="q" value="<?= h($q) ?>" placeholder="order id, email, payer, event title">
      </div>

      <div class="col-md-2 d-grid">
        <button class="btn btn-primary">Filter</button>
      </div>

    </form>

    <div class="row mt-3">
      <div class="col-md-4">
        <div class="p-2 bg-light rounded">
          <div class="text-muted small">Rows</div>
          <div class="fw-bold"><?= (int)$totalRows ?></div>
        </div>
      </div>
      <div class="col-md-4">
        <div class="p-2 bg-light rounded">
          <div class="text-muted small">Total Tickets (filtered)</div>
          <div class="fw-bold"><?= (int)($totals['total_tickets'] ?? 0) ?></div>
        </div>
      </div>
      <div class="col-md-4">
        <div class="p-2 bg-light rounded">
          <div class="text-muted small">Total Amount (filtered)</div>
          <div class="fw-bold">CAD <?= number_format((float)($totals['total_amount'] ?? 0), 2) ?></div>
        </div>
      </div>
    </div>
  </div>

  <div class="card p-0 overflow-hidden">
    <div class="table-responsive">
      <table class="table table-striped table-hover mb-0">
        <thead class="table-dark">
          <tr>
            <th>#</th>
            <th>Event</th>
            <th>Customer Email</th>
            <th>PayPal Email</th>
            <th>Payer</th>
            <th>Tickets</th>
            <th>Amount</th>
            <th>PayPal</th>
            <th>Status</th>
            <th>Order ID</th>
            <th>Date</th>
          </tr>
        </thead>
        <tbody>
        <?php if(empty($rows)): ?>
          <tr><td colspan="11" class="text-center py-4 text-muted">No bookings found</td></tr>
        <?php else: ?>
          <?php foreach($rows as $r): ?>
            <tr>
              <td class="fw-bold"><?= (int)$r['id'] ?></td>

              <td>
                <div class="fw-bold"><?= h($r['event_title']) ?></div>
                <div class="text-muted small">
                  <?php
                    $ts = !empty($r['event_date']) ? strtotime($r['event_date']) : 0;
                    echo $ts ? date("F d, Y g:i A", $ts) : '';
                  ?>
                </div>
              </td>

              <td><?= h($r['customer_email']) ?></td>
              <td><?= h($r['paypal_email']) ?></td>
              <td><?= h($r['payer_name']) ?></td>

              <td class="fw-bold"><?= (int)$r['ticket_count'] ?></td>

              <td class="fw-bold"><?= h($r['currency']) ?> <?= number_format((float)$r['amount'],2) ?></td>

              <td><?= paypalBadge($r['paypal_status']) ?></td>
              <td><?= badge($r['booking_status']) ?></td>

              <td class="mono"><?= h($r['order_id']) ?></td>

              <td>
                <div class="text-muted small"><?= h($r['created_at']) ?></div>
              </td>
            </tr>

            <!-- extra row: show ticket breakdown -->
            <tr>
              <td colspan="11" class="bg-light">
                <div class="d-flex justify-content-between flex-wrap gap-2">
                  <div>
                    <div class="text-muted small fw-bold">Ticket Breakdown</div>
                    <pre class="mb-0" style="white-space:pre-wrap;"><?= h($r['tickets_summary'] ?? '') ?></pre>
                  </div>
                  <div class="text-muted small">
                    <div><b>IP:</b> <?= h($r['ip_address']) ?></div>
                    <div><b>User Agent:</b> <?= h($r['user_agent']) ?></div>
                  </div>
                </div>
              </td>
            </tr>

          <?php endforeach; ?>
        <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>

  <!-- Pagination -->
  <nav class="mt-3">
    <ul class="pagination">
      <?php
        $qs = $_GET;
        for($p=1; $p<=$totalPages; $p++){
          $qs['page'] = $p;
          $url = "event_bookings.php?" . http_build_query($qs);
          $active = ($p === $page) ? "active" : "";
          echo '<li class="page-item '.$active.'"><a class="page-link" href="'.h($url).'">'.$p.'</a></li>';
        }
      ?>
    </ul>
  </nav>

</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
