<?php
require_once "db.php";

session_start();


// ✅ SECURITY CHECK 1: Must be logged in
if (!isset($_SESSION['user_id'])) {
  header("Location: index.php");
  exit;
}

// ✅ SECURITY CHECK 2: Must be admin
if (!isset($_SESSION['role']) || strtolower($_SESSION['role']) !== 'admin') {
  header("Location: index.php");
  exit;
}


header("Content-Type: application/json");

define("BATCH_SIZE", 20);
define("UNSUB_SECRET", "IAOS_UNSUB_2026_SECRET_KEY");

/* =========================
   ✅ Campaign ID
========================= */
$campaignId = isset($_GET['campaign_id']) ? (int)$_GET['campaign_id'] : 0;
if($campaignId <= 0){
  echo json_encode(["status"=>"error","message"=>"Invalid campaign id"]);
  exit;
}

/* =========================
   ✅ Settings + Campaign
========================= */
$set = $pdo->query("SELECT * FROM iaos_newsletter_settings WHERE id=1")->fetch(PDO::FETCH_ASSOC);
if(!$set){
  echo json_encode(["status"=>"error","message"=>"Newsletter settings missing"]);
  exit;
}

$camp = $pdo->prepare("SELECT * FROM newsletter_campaigns WHERE id=? LIMIT 1");
$camp->execute([$campaignId]);
$campaign = $camp->fetch(PDO::FETCH_ASSOC);

if(!$campaign){
  echo json_encode(["status"=>"error","message"=>"Campaign not found"]);
  exit;
}

/* =========================
   ✅ Helper: Public URL
========================= */
function publicUrl($path){
  $path = trim((string)$path);
  if($path === "") return "";

  // ✅ already full url
  if(preg_match('/^https?:\/\//i', $path)) return $path;

  // ✅ remove ../ and ..\
  $path = str_replace(["../", "..\\", "./", ".\\"], "", $path);
  $path = ltrim($path, "/\\");

  // ✅ remove admin_pannel/ if exists in DB
  $path = preg_replace("#^admin_pannel/#i", "", $path);
  $path = preg_replace("#^admin_pannel\\\\#i", "", $path);

  return "https://iaosregina.com/newweb/iaos/web/" . $path;
}

/* =========================
   ✅ Helper: Local path for CID
   Works for:
   - ../images/xxx.png
   - images/xxx.png
   - admin_pannel/../images/xxx.png
========================= */
function localPath($path){
  $path = trim((string)$path);
  if($path === "") return "";

  $path = str_replace(["../", "..\\", "./", ".\\"], "", $path);
  $path = ltrim($path, "/\\");

  // remove admin_pannel/ if exists
  $path = preg_replace("#^admin_pannel/#i", "", $path);
  $path = preg_replace("#^admin_pannel\\\\#i", "", $path);

  // ✅ Try 1: relative to web/admin_pannel/
  $try1 = realpath(__DIR__ . "/../" . $path);
  if($try1 && is_file($try1)) return $try1;

  // ✅ Try 2: relative to web/
  $try2 = realpath(__DIR__ . "/../../web/" . $path);
  if($try2 && is_file($try2)) return $try2;

  // ✅ Try 3: direct relative to current folder
  $try3 = realpath(__DIR__ . "/" . $path);
  if($try3 && is_file($try3)) return $try3;

  return "";
}

/* =========================
   ✅ Safe URL check
========================= */
function safeUrl($url){
  $url = trim((string)$url);
  if($url === "") return "";
  if(filter_var($url, FILTER_VALIDATE_URL)) return $url;
  return "";
}

/* =========================
   ✅ Send Mail With:
   ✅ Logo + Banner + Button
   ✅ Inline CID images if local exists
========================= */
function sendMailNow(
  $toEmail,
  $toName,
  $subject,
  $htmlBody,
  $fromEmail,
  $fromName,
  $replyTo,
  $img1 = "",
  $img2 = "",
  $logoLink = "",
  $bannerLink = "",
  $buttonLink = "",
  $buttonText = ""
){

  $toName  = trim($toName) ?: "Subscriber";
  $toEmail = trim($toEmail);

  // ✅ Replace template vars
  $htmlBody = str_replace(
    ["{name}","{email}"],
    [htmlspecialchars($toName), htmlspecialchars($toEmail)],
    $htmlBody
  );

  $token = sha1($toEmail . UNSUB_SECRET);
  $unsubLink = "https://iaosregina.com/unsubscribe.php?email=" . urlencode($toEmail) . "&token=" . $token;

  /* ✅ FIX LINKS */
  $logoLink   = safeUrl($logoLink) ?: "https://iaosregina.com/";
  $bannerLink = safeUrl($bannerLink) ?: $logoLink;
  $buttonLink = safeUrl($buttonLink) ?: $logoLink;

  $buttonText = trim((string)$buttonText);
  if($buttonText === "") $buttonText = "VISIT OUR WEBSITE";

  /* =========================
     ✅ LOGO as CID (BEST FIX)
  ========================= */
  $logoLocal = localPath("images/logo-white.png");
  $logoCID   = "logo_" . md5($toEmail . microtime(true));
  $logoUrl   = "https://iaosregina.com/newweb/iaos/web/images/logo-white.png";

  // if local exists => CID, else use URL
  $logoImg = $logoLocal ? ("cid:" . $logoCID) : $logoUrl;

  /* ✅ Prepare newsletter images */
  $img1Local = localPath($img1);
  $img2Local = localPath($img2);

  $img1Url = publicUrl($img1);
  $img2Url = publicUrl($img2);

  $cid1 = "img1_" . md5($toEmail . microtime(true));
  $cid2 = "img2_" . md5($toEmail . microtime(true) . "x");

  /* ✅ Image styles: CENTER + NO CROP */
  $imgStyle = "width:100%;max-width:640px;border-radius:12px;display:block;height:auto;margin:0 auto;";

  $bannerHtml = "";
  $imagesHtml = "";

  /* ✅ Banner (IMAGE 1 clickable) */
  if($img1Local){
    $bannerHtml = '
      <div style="margin:18px 0;text-align:center;">
        <a href="'.htmlspecialchars($bannerLink).'" target="_blank" style="text-decoration:none;">
          <img src="cid:'.$cid1.'" alt="Newsletter Banner" style="'.$imgStyle.'">
        </a>
      </div>
    ';
  } elseif($img1Url){
    $bannerHtml = '
      <div style="margin:18px 0;text-align:center;">
        <a href="'.htmlspecialchars($bannerLink).'" target="_blank" style="text-decoration:none;">
          <img src="'.htmlspecialchars($img1Url).'" alt="Newsletter Banner" style="'.$imgStyle.'">
        </a>
      </div>
    ';
  }

  /* ✅ Extra Image 2 (optional inside body) */
  if($img2Local){
    $imagesHtml .= '
      <div style="margin:18px 0;text-align:center;">
        <img src="cid:'.$cid2.'" alt="Newsletter Image" style="'.$imgStyle.'">
      </div>
    ';
  } elseif($img2Url){
    $imagesHtml .= '
      <div style="margin:18px 0;text-align:center;">
        <img src="'.htmlspecialchars($img2Url).'" alt="Newsletter Image" style="'.$imgStyle.'">
      </div>
    ';
  }

  /* ✅ CTA Button */
  $buttonHtml = '
    <div style="text-align:center;margin:24px 0;">
      <a href="'.htmlspecialchars($buttonLink).'" target="_blank"
         style="
            display:inline-block;
            background:#0b7b4c;
            color:#ffffff;
            text-decoration:none;
            font-weight:800;
            padding:14px 24px;
            border-radius:10px;
            font-size:14px;
         ">
        '.htmlspecialchars($buttonText).'
      </a>
    </div>
  ';

  /* ✅ Footer */
  $footer = '
    <hr style="border:none;border-top:1px solid #e5e7eb;margin:22px 0;">
    <div style="font-size:12px;color:#6b7280;line-height:1.6;">
      You received this email from IAOS Regina newsletter.<br>
      <a href="'.$unsubLink.'" style="color:#0b7b4c;font-weight:700;">Unsubscribe</a>
    </div>';

  /* ✅ FULL EMAIL TEMPLATE */
  $html = '
    <div style="background:#f4f6f8;padding:25px;font-family:Arial,Helvetica,sans-serif;">
      <div style="max-width:640px;margin:auto;background:#ffffff;border-radius:14px;overflow:hidden;box-shadow:0 8px 20px rgba(0,0,0,0.08);">

        <!-- ✅ TOP GREEN HEADER + LOGO -->
        <div style="background:#0b7b4c;padding:18px 22px;">
          <div style="text-align:center;">
            <a href="'.htmlspecialchars($logoLink).'" target="_blank" style="text-decoration:none;">
              <img src="'.htmlspecialchars($logoImg).'" alt="IAOS Regina"
                   style="height:44px;width:auto;display:inline-block;border:0;outline:none;text-decoration:none;">
            </a>
          </div>
        </div>

        <div style="padding:22px;font-size:14px;color:#111827;line-height:1.7;">

          <!-- ✅ Banner -->
          '.$bannerHtml.'

          <!-- ✅ Body -->
          '.$htmlBody.'

          <!-- ✅ Extra Image -->
          '.$imagesHtml.'

          <!-- ✅ Button -->
          '.$buttonHtml.'

          <!-- ✅ Footer -->
          '.$footer.'
        </div>

      </div>
    </div>
  ';

  /* ✅ plain text fallback */
  $plain  = strip_tags($htmlBody) . "\n\n";
  $plain .= "Link: ".$buttonLink."\n";
  $plain .= "Unsubscribe: ".$unsubLink;

  /* ✅ Boundaries */
  $boundaryMixed = "mix_" . md5(uniqid(time(), true));
  $boundaryAlt   = "alt_" . md5(uniqid(time(), true));

  $headers  = "From: {$fromName} <{$fromEmail}>\r\n";
  $headers .= "Reply-To: {$replyTo}\r\n";
  $headers .= "MIME-Version: 1.0\r\n";
  $headers .= "List-Unsubscribe: <{$unsubLink}>\r\n";
  $headers .= "Content-Type: multipart/related; boundary=\"{$boundaryMixed}\"\r\n";

  $body  = "--{$boundaryMixed}\r\n";
  $body .= "Content-Type: multipart/alternative; boundary=\"{$boundaryAlt}\"\r\n\r\n";

  // ✅ Plain
  $body .= "--{$boundaryAlt}\r\n";
  $body .= "Content-Type: text/plain; charset=UTF-8\r\n\r\n";
  $body .= $plain . "\r\n\r\n";

  // ✅ HTML
  $body .= "--{$boundaryAlt}\r\n";
  $body .= "Content-Type: text/html; charset=UTF-8\r\n\r\n";
  $body .= $html . "\r\n\r\n";

  $body .= "--{$boundaryAlt}--\r\n";

  /* =========================
     ✅ ATTACH CID IMAGES
  ========================= */

  /* ✅ Attach LOGO inline */
  if($logoLocal){
    $imgData = base64_encode(file_get_contents($logoLocal));
    $mime    = mime_content_type($logoLocal);
    $name    = basename($logoLocal);

    $body .= "--{$boundaryMixed}\r\n";
    $body .= "Content-Type: {$mime}; name=\"{$name}\"\r\n";
    $body .= "Content-Transfer-Encoding: base64\r\n";
    $body .= "Content-ID: <{$logoCID}>\r\n";
    $body .= "Content-Disposition: inline; filename=\"{$name}\"\r\n\r\n";
    $body .= chunk_split($imgData) . "\r\n";
  }

  /* ✅ Attach Banner Image 1 inline */
  if($img1Local){
    $imgData = base64_encode(file_get_contents($img1Local));
    $mime = mime_content_type($img1Local);

    $body .= "--{$boundaryMixed}\r\n";
    $body .= "Content-Type: {$mime}; name=\"banner\"\r\n";
    $body .= "Content-Transfer-Encoding: base64\r\n";
    $body .= "Content-ID: <{$cid1}>\r\n";
    $body .= "Content-Disposition: inline; filename=\"banner\"\r\n\r\n";
    $body .= chunk_split($imgData) . "\r\n";
  }

  /* ✅ Attach Image 2 inline */
  if($img2Local){
    $imgData = base64_encode(file_get_contents($img2Local));
    $mime = mime_content_type($img2Local);

    $body .= "--{$boundaryMixed}\r\n";
    $body .= "Content-Type: {$mime}; name=\"image2\"\r\n";
    $body .= "Content-Transfer-Encoding: base64\r\n";
    $body .= "Content-ID: <{$cid2}>\r\n";
    $body .= "Content-Disposition: inline; filename=\"image2\"\r\n\r\n";
    $body .= chunk_split($imgData) . "\r\n";
  }

  $body .= "--{$boundaryMixed}--\r\n";

  return mail($toEmail, $subject, $body, $headers, "-f" . $fromEmail);
}

/* ✅ Campaign content from DB */
$image1 = !empty($campaign['image1']) ? $campaign['image1'] : "";
$image2 = !empty($campaign['image2']) ? $campaign['image2'] : "";

/* ✅ NEW LINKS FROM DB */
$logoLink   = $campaign['logo_link'] ?? "";
$bannerLink = $campaign['banner_link'] ?? "";
$buttonLink = $campaign['button_link'] ?? "";
$buttonText = $campaign['button_text'] ?? "VISIT OUR WEBSITE";

/* ✅ Load pending batch */
$stmt = $pdo->prepare("
  SELECT nq.id as qid, ns.full_name, ns.email
  FROM newsletter_queue nq
  INNER JOIN newsletter_subscribers ns ON ns.id = nq.subscriber_id
  WHERE nq.campaign_id=? AND nq.status='pending' AND ns.status='active'
  ORDER BY nq.id ASC
  LIMIT ".BATCH_SIZE."
");
$stmt->execute([$campaignId]);
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* ✅ totals */
$total   = (int)$pdo->query("SELECT COUNT(*) FROM newsletter_queue WHERE campaign_id=".$campaignId)->fetchColumn();
$sent    = (int)$pdo->query("SELECT COUNT(*) FROM newsletter_queue WHERE campaign_id=".$campaignId." AND status='sent'")->fetchColumn();
$pending = (int)$pdo->query("SELECT COUNT(*) FROM newsletter_queue WHERE campaign_id=".$campaignId." AND status='pending'")->fetchColumn();

if(empty($rows)){
  $percent = ($total > 0) ? round(($sent/$total)*100) : 100;
  echo json_encode(["status"=>"done","total"=>$total,"sent"=>$sent,"pending"=>0,"percent"=>$percent]);
  exit;
}

$updSent = $pdo->prepare("UPDATE newsletter_queue SET status='sent', sent_at=NOW() WHERE id=? LIMIT 1");
$updFail = $pdo->prepare("UPDATE newsletter_queue SET status='failed', error_message=? WHERE id=? LIMIT 1");

foreach($rows as $r){
  $toEmail = trim($r['email']);
  $toName  = trim($r['full_name'] ?? "Subscriber");

  $ok = sendMailNow(
    $toEmail,
    $toName,
    $campaign['subject'],
    $campaign['html_body'],
    $set['from_email'],
    $set['from_name'],
    ($set['reply_to'] ?: $set['from_email']),
    $image1,
    $image2,
    $logoLink,
    $bannerLink,
    $buttonLink,
    $buttonText
  );

  if($ok){
    $updSent->execute([(int)$r['qid']]);
  } else {
    $updFail->execute(["mail() failed", (int)$r['qid']]);
  }
}

/* ✅ return progress */
$sent    = (int)$pdo->query("SELECT COUNT(*) FROM newsletter_queue WHERE campaign_id=".$campaignId." AND status='sent'")->fetchColumn();
$pending = (int)$pdo->query("SELECT COUNT(*) FROM newsletter_queue WHERE campaign_id=".$campaignId." AND status='pending'")->fetchColumn();
$percent = ($total > 0) ? round(($sent/$total)*100) : 0;

echo json_encode([
  "status"=>"sending",
  "total"=>$total,
  "sent"=>$sent,
  "pending"=>$pending,
  "percent"=>$percent
]);
exit;
