<?php
/****************************************************
 * 1 PAGE PAYPAL ONE-TIME PAYMENT (SMART BUTTONS)
 * FILE: paypal_payment.php
 * ✅ Frontend button + Backend capture in SAME file
 ****************************************************/

// -------------------- SETTINGS --------------------
$PAYPAL_MODE = "live"; // "sandbox" OR "live"

$PAYPAL_CLIENT_ID = "";
$PAYPAL_SECRET_ID = "";

// ✅ Payment Amount
$PAY_AMOUNT   = "1.00"; // example
$PAY_CURRENCY = "CAD";   // USD / CAD / etc

// -------------------- PAYPAL API BASE URL --------------------
$baseUrl = ($PAYPAL_MODE === "live")
    ? "https://api-m.paypal.com"
    : "https://api-m.sandbox.paypal.com";


// ============================================================
// ✅ AJAX REQUEST: CAPTURE ORDER (SERVER SIDE)
// ============================================================
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_GET["action"]) && $_GET["action"] === "capture") {

    header("Content-Type: application/json");

    $input = json_decode(file_get_contents("php://input"), true);
    $orderID = $input["orderID"] ?? "";

    if (!$orderID) {
        echo json_encode(["status" => "error", "message" => "Missing orderID"]);
        exit;
    }

    // ✅ 1) Get Access Token
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $baseUrl . "/v1/oauth2/token");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_USERPWD, $PAYPAL_CLIENT_ID . ":" . $PAYPAL_SECRET_ID);
    curl_setopt($ch, CURLOPT_POSTFIELDS, "grant_type=client_credentials");
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Accept: application/json",
        "Accept-Language: en_US"
    ]);

    $tokenResponse = curl_exec($ch);
    curl_close($ch);

    $tokenData = json_decode($tokenResponse, true);
    $accessToken = $tokenData["access_token"] ?? "";

    if (!$accessToken) {
        echo json_encode([
            "status" => "error",
            "message" => "Failed to get PayPal access token",
            "debug" => $tokenData
        ]);
        exit;
    }

    // ✅ 2) Capture Payment
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $baseUrl . "/v2/checkout/orders/" . $orderID . "/capture");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Content-Type: application/json",
        "Authorization: Bearer " . $accessToken
    ]);

    $captureResponse = curl_exec($ch);
    curl_close($ch);

    $captureData = json_decode($captureResponse, true);

    // ✅ Check if completed
    $status = $captureData["status"] ?? "";

    if ($status === "COMPLETED") {

        $payerEmail = $captureData["payer"]["email_address"] ?? "";
        $payerName  = ($captureData["payer"]["name"]["given_name"] ?? "") . " " . ($captureData["payer"]["name"]["surname"] ?? "");

        $amountValue = $captureData["purchase_units"][0]["payments"]["captures"][0]["amount"]["value"] ?? "";
        $currency    = $captureData["purchase_units"][0]["payments"]["captures"][0]["amount"]["currency_code"] ?? "";

        // ✅ OPTIONAL: Save to your database here
        // Example: orderID, payerEmail, amountValue, status, created_at

        echo json_encode([
            "status" => "success",
            "message" => "Payment Captured ✅",
            "orderID" => $orderID,
            "payerEmail" => $payerEmail,
            "payerName" => trim($payerName),
            "amount" => $amountValue,
            "currency" => $currency
        ]);
        exit;
    }

    echo json_encode([
        "status" => "error",
        "message" => "Payment not completed",
        "paypal_status" => $status,
        "debug" => $captureData
    ]);
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>PayPal One-Time Payment</title>

  <style>
    body{
      font-family: Arial, sans-serif;
      background:#f5f5f5;
      padding:40px;
    }
    .box{
      max-width:520px;
      margin:auto;
      background:#fff;
      padding:25px;
      border-radius:12px;
      box-shadow:0 6px 18px rgba(0,0,0,0.08);
      text-align:center;
    }
    .price{
      font-size:24px;
      font-weight:700;
      margin:10px 0 20px;
    }
    #result{
      margin-top:20px;
      padding:12px;
      border-radius:10px;
      display:none;
      text-align:left;
      font-size:14px;
      line-height:1.5;
    }
    .success{ background:#d1f7d6; color:#0f5132; }
    .error{ background:#f8d7da; color:#842029; }
  </style>
</head>

<body>

<div class="box">
  <h2>PayPal One-Time Payment</h2>
  <div class="price">
    Pay: <?= htmlspecialchars($PAY_CURRENCY) ?> <?= htmlspecialchars($PAY_AMOUNT) ?>
  </div>

  <div id="paypal-button-container"></div>

  <div id="result"></div>
</div>

<!-- ✅ PAYPAL SDK (ONE TIME PAYMENT) -->
<script src="https://www.paypal.com/sdk/js?client-id=<?= htmlspecialchars($PAYPAL_CLIENT_ID) ?>&currency=<?= htmlspecialchars($PAY_CURRENCY) ?>"></script>

<script>
const resultBox = document.getElementById("result");

// ✅ PayPal Button
paypal.Buttons({

  createOrder: function(data, actions) {
    return actions.order.create({
      purchase_units: [{
        amount: {
          value: "<?= $PAY_AMOUNT ?>"
        }
      }]
    });
  },

  onApprove: function(data, actions) {

    // ✅ IMPORTANT: capture on YOUR SERVER (secure)
    return fetch("<?= basename(__FILE__) ?>?action=capture", {
      method: "POST",
      headers: {
        "Content-Type": "application/json"
      },
      body: JSON.stringify({
        orderID: data.orderID
      })
    })
    .then(res => res.json())
    .then(resp => {
      resultBox.style.display = "block";

      if(resp.status === "success"){
        resultBox.className = "success";
        resultBox.innerHTML = `
          <b>✅ Payment Success!</b><br>
          <b>Order ID:</b> ${resp.orderID}<br>
          <b>Name:</b> ${resp.payerName}<br>
          <b>Email:</b> ${resp.payerEmail}<br>
          <b>Paid:</b> ${resp.currency} ${resp.amount}<br>
        `;
      } else {
        resultBox.className = "error";
        resultBox.innerHTML = `
          <b>❌ Payment Failed</b><br>
          <b>Message:</b> ${resp.message}<br>
          <pre style="white-space:pre-wrap;">${JSON.stringify(resp.debug || resp, null, 2)}</pre>
        `;
      }
    })
    .catch(err => {
      resultBox.style.display = "block";
      resultBox.className = "error";
      resultBox.innerHTML = "❌ Server Error: " + err;
      console.error(err);
    });

  },

  onCancel: function () {
    resultBox.style.display = "block";
    resultBox.className = "error";
    resultBox.innerHTML = "⚠️ Payment cancelled by user.";
  },

  onError: function (err) {
    resultBox.style.display = "block";
    resultBox.className = "error";
    resultBox.innerHTML = "❌ PayPal Error: " + err;
    console.error(err);
  }

}).render("#paypal-button-container");
</script>

</body>
</html>
