<?php
/*************************************************************
 * selectedevent.php
 * ✅ Event Details Page (Like Reference Screenshot)
 * ✅ Tickets + / - (add remove)
 * ✅ One-Time PayPal Payment (Smart Buttons)
 * ✅ Member Ticket visible only when member logged in
 * ✅ Extra ticket options 1..5 from events table
 * ✅ Send email on successful payment
 * ✅ PayPal button render fixed (renders only when user clicks Pay)
 * ✅ NEW: Take email before Pay/Register and send ticket details to that email
 *************************************************************/

session_start();

/* -----------------------------------------
   0) MAIN DB CONNECTIONS
------------------------------------------ */
require_once "admin_pannel/db.php"; // gives $pdo (your admin panel DB)
require_once "../db.php";           // gives get_db()

$pdoEvents = get_db();
$pdoEvents->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

/* -----------------------------------------
   ✅ MEMBER LOGIN CHECK
------------------------------------------ */
$isMemberLoggedIn = false;
if (!empty($_SESSION['userType']) && strtolower($_SESSION['userType']) === 'member') {
    $isMemberLoggedIn = true;
}

/* -----------------------------------------
   5) PAYPAL CONFIG (ONE TIME PAYMENT)
------------------------------------------ */
$PAYPAL_MODE = "live"; // "sandbox" OR "live"

// ✅ Put your real keys here
$PAYPAL_CLIENT_ID = "AV_riSlunWTn1mKi6Xen1m2R-EvAxb55Wni-3nNThnyyPMMah9qd5dgEE6CHvalVcrcLxR6w1oa9ReJA";
$PAYPAL_SECRET_ID = "EBhrsEZ1dT4YbS5i6xVhVJU8ws5bHqlFCnl0A43AZcsS5ZoAOdYSajLRn5GzVTFQ3_uoFOXt7rhaes8_";

// ✅ Currency
$PAY_CURRENCY = "CAD";

// ✅ PayPal API BASE URL
$baseUrl = ($PAYPAL_MODE === "live")
    ? "https://api-m.paypal.com"
    : "https://api-m.sandbox.paypal.com";


/* -----------------------------------------
   ✅ Helper: Build tickets list from event row
   ✅ THIS WAS MISSING (IMPORTANT)
------------------------------------------ */
function buildTicketsFromEvent($eventRow, $isMemberLoggedIn){
    $tickets = [];

    // ✅ Adult Admission
    $adultPrice = (float)($eventRow['base_price'] ?? 0);
    if ($adultPrice > 0) {
        $tickets[] = [
            "key"   => "adult",
            "name"  => "Adult Admission",
            "price" => $adultPrice
        ];
    }

    // ✅ Member Admission only when member logged in
    $memberPrice = (float)($eventRow['member_price'] ?? 0);
    if ($isMemberLoggedIn && $memberPrice > 0) {
        $tickets[] = [
            "key"   => "member",
            "name"  => "Member Admission",
            "price" => $memberPrice
        ];
    }

    // ✅ Extra ticket options 1..5
    for($i=1; $i<=5; $i++){
        $h = trim($eventRow["ticketoption{$i}heading"] ?? "");
        $p = (float)($eventRow["ticketoption{$i}price"] ?? 0);

        if($h !== "" && $p > 0){
            $tickets[] = [
                "key"   => "opt".$i,
                "name"  => $h,
                "price" => $p
            ];
        }
    }

    return $tickets;
}


/* -----------------------------------------
   ✅ BEAUTIFUL EMAIL FUNCTION (HTML + TEXT)
------------------------------------------ */
function sendReceiptEmail(
    $toEmail,
    $eventTitle,
    $orderID,
    $totalPaid,
    $currency,
    $ticketsSummary,
    $eventDateTxt = "",
    $eventTimeTxt = "",
    $eventLocation = ""
){
    if(empty($toEmail)) return false;

    // ✅ IMPORTANT: Use YOUR DOMAIN email as From (same domain)
    $fromEmail = "testuser@iaosregina.com";
    $fromName  = "IAOS Regina";
    $replyTo   = "info@iaosregina.com";

    // ✅ Subject WITHOUT emoji (emoji makes Gmail spam sometimes)
    $subject = "IAOS Regina - Ticket Confirmation (Order " . $orderID . ")";

    // ✅ Convert ticket summary text -> HTML list
    $ticketLines = array_filter(array_map("trim", explode("\n", $ticketsSummary)));
    $ticketHtml = "";
    foreach($ticketLines as $line){
        $ticketHtml .= "<li style='margin:6px 0;color:#111827;font-size:14px;'>" . htmlspecialchars($line) . "</li>";
    }
    if($ticketHtml === "") $ticketHtml = "<li>No ticket details found</li>";

    // ✅ Email HTML Body (Beautiful)
    $html = '
    <div style="background:#f4f6f8;padding:25px;font-family:Arial,Helvetica,sans-serif;">
      <div style="max-width:640px;margin:auto;background:#ffffff;border-radius:14px;overflow:hidden;box-shadow:0 8px 20px rgba(0,0,0,0.08);">

        <div style="background:#0b7b4c;padding:18px 22px;">
          <div style="color:#ffffff;font-size:20px;font-weight:800;">IAOS Regina</div>
          <div style="color:#eafff3;font-size:13px;margin-top:4px;">Ticket Confirmation</div>
        </div>

        <div style="padding:22px;">
          <div style="font-size:15px;color:#111827;line-height:1.6;">
            <b>Assalamu Alaikum,</b><br>
            JazakAllahu Khairan for your payment. Your booking has been confirmed successfully ✅
          </div>

          <div style="margin-top:16px;padding:14px;border:1px solid #e5e7eb;border-radius:12px;background:#f9fafb;">
            <div style="font-size:13px;color:#6b7280;font-weight:700;text-transform:uppercase;letter-spacing:0.6px;">Order Details</div>

            <table style="width:100%;margin-top:10px;border-collapse:collapse;font-size:14px;">
              <tr>
                <td style="padding:8px 0;color:#374151;width:170px;"><b>Order ID:</b></td>
                <td style="padding:8px 0;color:#111827;">'.htmlspecialchars($orderID).'</td>
              </tr>
              <tr>
                <td style="padding:8px 0;color:#374151;"><b>Event:</b></td>
                <td style="padding:8px 0;color:#111827;">'.htmlspecialchars($eventTitle).'</td>
              </tr>';

    if($eventDateTxt !== "" || $eventTimeTxt !== ""){
        $html .= '
              <tr>
                <td style="padding:8px 0;color:#374151;"><b>Date & Time:</b></td>
                <td style="padding:8px 0;color:#111827;">'.htmlspecialchars(trim($eventDateTxt." ".$eventTimeTxt)).'</td>
              </tr>';
    }

    if($eventLocation !== ""){
        $html .= '
              <tr>
                <td style="padding:8px 0;color:#374151;"><b>Location:</b></td>
                <td style="padding:8px 0;color:#111827;">'.htmlspecialchars($eventLocation).'</td>
              </tr>';
    }

    $html .= '
              <tr>
                <td style="padding:8px 0;color:#374151;"><b>Paid Amount:</b></td>
                <td style="padding:8px 0;color:#111827;"><b>'.htmlspecialchars($currency)." ".htmlspecialchars($totalPaid).'</b></td>
              </tr>
            </table>
          </div>

          <div style="margin-top:16px;padding:14px;border:1px solid #e5e7eb;border-radius:12px;">
            <div style="font-size:13px;color:#6b7280;font-weight:700;text-transform:uppercase;letter-spacing:0.6px;">Tickets Purchased</div>
            <ul style="margin:10px 0 0 18px;padding:0;">
              '.$ticketHtml.'
            </ul>
          </div>

          <div style="margin-top:18px;padding:14px;border-radius:12px;background:#ecfdf5;border:1px solid #bbf7d0;">
            <div style="color:#065f46;font-weight:800;">✅ Booking Confirmed</div>
            <div style="color:#065f46;font-size:13px;margin-top:4px;">
              Please keep this email as your receipt. If you have any questions, reply to this email.
            </div>
          </div>

          <div style="margin-top:18px;color:#6b7280;font-size:13px;line-height:1.5;">
            Regards,<br>
            <b>IAOS Regina Team</b><br>
            <span style="color:#111827;">Website:</span> https://iaosregina.com/
          </div>
        </div>

        <div style="background:#111827;color:#9ca3af;padding:14px 22px;font-size:12px;">
          This is an automated email. Please do not share your payment details with anyone.
        </div>

      </div>
    </div>';

    // ✅ Plain text fallback (important for inbox)
    $plain = "Assalamu Alaikum,\n\n";
    $plain .= "Thank you for your payment. Your booking is confirmed.\n\n";
    $plain .= "Order ID: ".$orderID."\n";
    $plain .= "Event: ".$eventTitle."\n";
    if($eventDateTxt || $eventTimeTxt) $plain .= "Date & Time: ".trim($eventDateTxt." ".$eventTimeTxt)."\n";
    if($eventLocation) $plain .= "Location: ".$eventLocation."\n";
    $plain .= "Paid Amount: ".$currency." ".$totalPaid."\n\n";
    $plain .= "Tickets:\n".$ticketsSummary."\n";
    $plain .= "\nRegards,\nIAOS Regina\nhttps://iaosregina.com/\n";

    // ✅ Build multipart email (HTML + Text)
    $boundary = "bnd_" . md5(uniqid(time(), true));

    $headers  = "From: {$fromName} <{$fromEmail}>\r\n";
    $headers .= "Reply-To: {$replyTo}\r\n";
    $headers .= "MIME-Version: 1.0\r\n";
    $headers .= "Message-ID: <" . md5(uniqid()) . "@iaosregina.com>\r\n";
    $headers .= "X-Mailer: PHP/" . phpversion() . "\r\n";
    $headers .= "Content-Type: multipart/alternative; boundary=\"{$boundary}\"\r\n";

    $body  = "--{$boundary}\r\n";
    $body .= "Content-Type: text/plain; charset=UTF-8\r\n";
    $body .= "Content-Transfer-Encoding: 7bit\r\n\r\n";
    $body .= $plain . "\r\n\r\n";

    $body .= "--{$boundary}\r\n";
    $body .= "Content-Type: text/html; charset=UTF-8\r\n";
    $body .= "Content-Transfer-Encoding: 7bit\r\n\r\n";
    $body .= $html . "\r\n\r\n";

    $body .= "--{$boundary}--\r\n";

    // ✅ VERY IMPORTANT: Set Envelope Sender (-f) helps Gmail inbox a lot
    return mail($toEmail, $subject, $body, $headers, "-f" . $fromEmail);
}


/* -----------------------------------------
   ✅ AJAX CAPTURE PAYMENT (SERVER SIDE)
------------------------------------------ */
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_GET["action"]) && $_GET["action"] === "capture") {

    header("Content-Type: application/json");

    $input = json_decode(file_get_contents("php://input"), true);

    $orderID     = $input["orderID"] ?? "";
    $eventIDPost = (int)($input["eventID"] ?? 0);
    $totalPosted = (float)($input["total"] ?? 0);
    $qtyPosted   = $input["qty"] ?? [];

    // ✅ NEW: email from user input before pay
    $customerEmailRaw = trim($input["customer_email"] ?? "");
    $customerEmail = filter_var($customerEmailRaw, FILTER_VALIDATE_EMAIL) ? $customerEmailRaw : "";

    if (!$orderID || $eventIDPost <= 0) {
        echo json_encode(["status" => "error", "message" => "Missing orderID / eventID"]);
        exit;
    }

    if (!is_array($qtyPosted) || empty($qtyPosted)) {
        echo json_encode(["status" => "error", "message" => "No ticket data received"]);
        exit;
    }

    // ✅ total tickets count
    $ticketCount = 0;
    foreach($qtyPosted as $k => $v) {
        $ticketCount += (int)$v;
    }

    if ($ticketCount <= 0) {
        echo json_encode(["status" => "error", "message" => "No tickets selected"]);
        exit;
    }

    // ✅ Must have valid email
    if (empty($customerEmail)) {
        echo json_encode(["status" => "error", "message" => "Customer email is required"]);
        exit;
    }

    /* ✅ Re-check event info live from DB (UPDATED: subtitle + event_date added) */
    $check = $pdoEvents->prepare("
        SELECT
          id, title, subtitle, event_date, capacity, booked, status,
          base_price, member_price,

          ticketoption1heading, ticketoption1price,
          ticketoption2heading, ticketoption2price,
          ticketoption3heading, ticketoption3price,
          ticketoption4heading, ticketoption4price,
          ticketoption5heading, ticketoption5price

        FROM events
        WHERE id = ?
        LIMIT 1
    ");
    $check->execute([$eventIDPost]);
    $ev = $check->fetch(PDO::FETCH_ASSOC);

    if(!$ev){
        echo json_encode(["status" => "error", "message" => "Event not found"]);
        exit;
    }

    $cap   = (int)$ev['capacity'];
    $bkd   = (int)$ev['booked'];
    $avail = max(0, $cap - $bkd);

    if (strtolower(trim($ev['status'])) !== 'open' || $avail <= 0) {
        echo json_encode(["status" => "error", "message" => "Event is sold out"]);
        exit;
    }

    if ($ticketCount > $avail) {
        echo json_encode(["status" => "error", "message" => "Not enough seats available"]);
        exit;
    }

    /* ✅ Build ticket list + price map from DB (secure) */
    $ticketsList = buildTicketsFromEvent($ev, $isMemberLoggedIn);
    $priceMap = [];
    $nameMap  = [];

    foreach($ticketsList as $t){
        $priceMap[$t['key']] = (float)$t['price'];
        $nameMap[$t['key']]  = $t['name'];
    }

    /* ✅ Calculate total from DB prices (secure) */
    $serverTotal = 0;
    foreach($qtyPosted as $k => $v){
        $v = (int)$v;
        if($v <= 0) continue;

        if(!isset($priceMap[$k])){
            echo json_encode([
                "status" => "error",
                "message" => "Invalid ticket type selected: " . $k
            ]);
            exit;
        }

        $serverTotal += ($v * $priceMap[$k]);
    }

    $serverTotal = round($serverTotal, 2);

    if ($serverTotal <= 0) {
        echo json_encode(["status" => "error", "message" => "Total is zero"]);
        exit;
    }

    // ✅ Compare posted total with server total (strict)
    if (abs($serverTotal - $totalPosted) > 0.02) {
        echo json_encode([
            "status" => "error",
            "message" => "Total mismatch (security check failed)",
            "debug" => [
                "posted_total" => $totalPosted,
                "server_total" => $serverTotal
            ]
        ]);
        exit;
    }

    /* ✅ 1) Get PayPal Access Token */
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $baseUrl . "/v1/oauth2/token");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_USERPWD, $PAYPAL_CLIENT_ID . ":" . $PAYPAL_SECRET_ID);
    curl_setopt($ch, CURLOPT_POSTFIELDS, "grant_type=client_credentials");
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Accept: application/json",
        "Accept-Language: en_US"
    ]);

    $tokenResponse = curl_exec($ch);
    curl_close($ch);

    $tokenData = json_decode($tokenResponse, true);
    $accessToken = $tokenData["access_token"] ?? "";

    if (!$accessToken) {
        echo json_encode([
            "status" => "error",
            "message" => "Failed to get PayPal access token",
            "debug" => $tokenData
        ]);
        exit;
    }

    /* ✅ 2) Capture Payment */
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $baseUrl . "/v2/checkout/orders/" . $orderID . "/capture");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Content-Type: application/json",
        "Authorization: Bearer " . $accessToken
    ]);

    $captureResponse = curl_exec($ch);
    curl_close($ch);

    $captureData = json_decode($captureResponse, true);
    $statusPay = $captureData["status"] ?? "";

    if ($statusPay === "COMPLETED") {

        $payerEmail = $captureData["payer"]["email_address"] ?? "";
        $payerName  = ($captureData["payer"]["name"]["given_name"] ?? "") . " " . ($captureData["payer"]["name"]["surname"] ?? "");

        $paidValue = $captureData["purchase_units"][0]["payments"]["captures"][0]["amount"]["value"] ?? "0";
        $currency  = $captureData["purchase_units"][0]["payments"]["captures"][0]["amount"]["currency_code"] ?? "";

        // ✅ final amount check
        if (abs((float)$paidValue - (float)$serverTotal) > 0.02) {
            echo json_encode([
                "status" => "error",
                "message" => "Payment amount mismatch",
                "debug" => [
                    "server_total" => $serverTotal,
                    "paid_total"   => $paidValue
                ]
            ]);
            exit;
        }

        /* ✅ 3) Update booked seats */
        $upd = $pdoEvents->prepare("UPDATE events SET booked = booked + ? WHERE id = ? LIMIT 1");
        $upd->execute([$ticketCount, $eventIDPost]);

        /* ✅ 4) Create nice ticket summary */
        $ticketsSummary = "";
        foreach($qtyPosted as $k => $v){
            $v = (int)$v;
            if($v > 0){
                $ticketName = $nameMap[$k] ?? strtoupper($k);
                $ticketsSummary .= "- " . $ticketName . " x " . $v . "\n";
            }
        }

        // ✅ Date/Time formatting for email
        $eventDateTxtEmail = "";
        $eventTimeTxtEmail = "";
        if(!empty($ev['event_date'])){
            $ts = strtotime($ev['event_date']);
            $eventDateTxtEmail = date("F d, Y", $ts);
            $eventTimeTxtEmail = date("g:i A", $ts);
        }

        /* ✅ 5) Send receipt email to USER INPUT EMAIL (important) */
        $emailSent = sendReceiptEmail(
            $customerEmail,
            $ev['title'],
            $orderID,
            $paidValue,
            $currency,
            $ticketsSummary,
            $eventDateTxtEmail,
            $eventTimeTxtEmail,
            $ev['subtitle'] ?? ""
        );

        echo json_encode([
            "status" => "success",
            "message" => "Payment Captured ✅",
            "orderID" => $orderID,
            "eventID" => $eventIDPost,

            // ✅ email info
            "customerEmail" => $customerEmail,
            "paypalEmail"   => $payerEmail,

            "payerName" => trim($payerName),
            "amount" => $paidValue,
            "currency" => $currency,
            "email_sent" => $emailSent ? "yes" : "no"
        ]);
        exit;
    }

    echo json_encode([
        "status" => "error",
        "message" => "Payment not completed",
        "paypal_status" => $statusPay,
        "debug" => $captureData
    ]);
    exit;
}


/* -----------------------------------------
   1) GET EVENT ID (PAGE LOAD)
------------------------------------------ */
$eventID = isset($_GET['eventID']) ? (int)$_GET['eventID'] : 0;
if ($eventID <= 0) {
    die("<h3 style='padding:30px;color:red;'>Invalid Event ID</h3>");
}

/* -----------------------------------------
   2) FETCH EVENT DATA (PAGE LOAD)
------------------------------------------ */
$stmt = $pdoEvents->prepare("
    SELECT id, title, subtitle, event_date, expiry_date, image, capacity, booked, status,
           member_discount, created_at, base_price, member_price, link, description,

           ticketoption1heading, ticketoption1price,
           ticketoption2heading, ticketoption2price,
           ticketoption3heading, ticketoption3price,
           ticketoption4heading, ticketoption4price,
           ticketoption5heading, ticketoption5price

    FROM events
    WHERE id = ?
    LIMIT 1
");
$stmt->execute([$eventID]);
$event = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$event) {
    die("<h3 style='padding:30px;color:red;'>Event not found</h3>");
}

/* -----------------------------------------
   3) FETCH EXTRA PARAGRAPHS
------------------------------------------ */
$paraStmt = $pdoEvents->prepare("
    SELECT paragraph_text
    FROM event_paragraphs
    WHERE event_id = ?
    ORDER BY id ASC
");
$paraStmt->execute([$eventID]);
$eventParagraphs = $paraStmt->fetchAll(PDO::FETCH_ASSOC);

/* -----------------------------------------
   4) EVENT STATUS + AVAILABLE SEATS
------------------------------------------ */
$status = strtolower(trim($event['status']));
$isOpen = ($status === "open");

$capacity = (int)($event['capacity'] ?? 0);
$booked   = (int)($event['booked'] ?? 0);
$availableSeats = max(0, $capacity - $booked);

$badgeClass = $isOpen ? "bg-success" : "bg-danger";
$badgeText  = $isOpen ? "Open" : "Sold Out";

/* -----------------------------------------
   6) FORMAT DATE/TIME
------------------------------------------ */
$eventDateFull = strtotime($event['event_date']);
$eventDayName  = date("l", $eventDateFull);
$eventDateTxt  = date("F d, Y", $eventDateFull);
$eventTimeTxt  = date("g:i A", $eventDateFull);

/* -----------------------------------------
   7) BUILD TICKETS LIST (PAGE)
------------------------------------------ */
$tickets = buildTicketsFromEvent($event, $isMemberLoggedIn);
$memberPriceForMsg = (float)($event['member_price'] ?? 0);

?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title><?= htmlspecialchars($event['title']) ?> | IAOS Regina</title>

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
  <link rel="stylesheet" href="css/style.css">
  <link rel="icon" type="image/png" href="images/fav.png">

  <style>
    body{ background:#f4f6f8; }
    .page-title{ font-size:42px;font-weight:300;margin:30px 0 20px;color:#1f2937; }
    .top-card{ background:#fff;border:1px solid #e5e7eb;border-radius:10px;overflow:hidden;box-shadow:0 6px 18px rgba(0,0,0,0.05); }
    .event-banner{
  width:50%;
  height:100%;
  background:#0b2a4a;
  background-size:cover;
  background-position:center;
  text-align:center;

  margin:0 auto;
  display:flex;
  align-items:center;
  justify-content:center;
}
    .tickets-box{ border-left:1px solid #e5e7eb;padding:18px;height:100%;background:#fafafa; }
    .tickets-head{ font-size:12px;font-weight:800;letter-spacing:1px;color:#6b7280;text-transform:uppercase;margin:12px 0 10px; }
    .ticket-row{ display:flex;justify-content:space-between;align-items:center;padding:12px 0;border-top:1px solid #e5e7eb; }
    .ticket-row:first-child{ border-top:none; }
    .ticket-name{ font-weight:600;color:#111827; }
    .ticket-price{ font-weight:700;color:#111827;min-width:90px;text-align:right; }
    .qty-box{ display:flex;align-items:center;gap:6px;margin-left:14px; }
    .qty-btn{ width:34px;height:34px;border-radius:7px;border:1px solid #d1d5db;background:#fff;font-size:18px;font-weight:800;cursor:pointer; }
    .qty-btn:hover{ background:#f3f4f6; }
    .qty-input{ width:48px;height:34px;border:1px solid #d1d5db;border-radius:7px;text-align:center;font-weight:800;background:#fff;outline:none; }
    .register-btn{ width:100%;border:none;background:#5da6d9;color:#fff;font-weight:800;padding:10px 14px;border-radius:7px; }
    .register-btn:hover{ background:#438ec4; }
    .total-line{ display:flex;justify-content:space-between;align-items:center;border-top:1px solid #e5e7eb;margin-top:12px;padding-top:12px;font-weight:800; }
    .content-card{ background:#fff;border:1px solid #e5e7eb;border-radius:10px;box-shadow:0 6px 18px rgba(0,0,0,0.05);overflow:hidden; }
    .left-desc{ padding:22px; }
    .left-desc h5{ font-weight:800;margin-bottom:12px; }
    .sidebar{ border-left:1px solid #e5e7eb;padding:20px;background:#fafafa;height:100%; }
    .sidebar h6{ font-size:12px;font-weight:800;color:#6b7280;text-transform:uppercase;margin-bottom:10px; }
    .sidebar .block{ padding:12px 0;border-bottom:1px solid #e5e7eb; }
    .sidebar .block:last-child{ border-bottom:none; }
    #resultBox{ display:none;margin-top:15px;padding:12px;border-radius:10px;font-size:14px;line-height:1.5; }
    .successBox{ background:#d1f7d6;color:#0f5132; }
    .errorBox{ background:#f8d7da;color:#842029; }
    @media (max-width: 992px){
      .tickets-box{ border-left:none;border-top:1px solid #e5e7eb; }
      .sidebar{ border-left:none;border-top:1px solid #e5e7eb; }
      .page-title{ font-size:30px; }
      .event-banner{
    width:100%;   /* ✅ no cropping because container not too small */
  }
    }
  </style>
</head>
<body>

<div class="prayer-btn">
  <img src="images/btn_prayer.png" alt="Prayer Times" data-bs-toggle="modal" data-bs-target="#prayerModal">
</div>

<?php include 'header.php'; ?>

<div class="container-xl">

  <div class="page-title">
    <?= htmlspecialchars($event['title']) ?>
  </div>

  <div class="top-card mb-4">
    <div class="row g-0">

      <div class="col-lg-8">
        <div class="event-banner" style="background-image:url('../<?= htmlspecialchars($event['image']) ?>');"></div>
      </div>

      <div class="col-lg-4">
        <div class="tickets-box">

          <div class="d-flex justify-content-between align-items-center">
            <div>
              <div class="text-muted small"><?= htmlspecialchars($eventDayName) ?></div>
              <div class="fw-bold"><?= htmlspecialchars($eventDateTxt) ?></div>
            </div>

            <span class="badge <?= $badgeClass ?> px-3 py-2"><?= $badgeText ?></span>
          </div>

          <div class="tickets-head">Tickets</div>

          <?php foreach($tickets as $t): ?>
            <div class="ticket-row">
              <div class="ticket-name"><?= htmlspecialchars($t['name']) ?></div>

              <div class="d-flex align-items-center">
                <div class="ticket-price">$<?= number_format((float)$t['price'], 2) ?></div>

                <div class="qty-box">
                  <button class="qty-btn minus" data-key="<?= htmlspecialchars($t['key']) ?>">-</button>
                  <input class="qty-input" id="qty_<?= htmlspecialchars($t['key']) ?>" value="0" readonly>
                  <button class="qty-btn plus" data-key="<?= htmlspecialchars($t['key']) ?>">+</button>
                </div>
              </div>
            </div>
          <?php endforeach; ?>

          <?php if(!$isMemberLoggedIn && $memberPriceForMsg > 0): ?>
            <div class="text-muted small mt-2">
              ✅ Login as Member to get Member Admission ticket.
            </div>
          <?php endif; ?>

          <div class="total-line">
            <span>Total</span>
            <span>$ <span id="totalAmount">0.00</span></span>
          </div>

          <!-- ✅ NEW EMAIL FIELD BEFORE PAY -->
          <div class="mt-3">
            <label class="form-label fw-bold" style="font-size:13px;color:#374151;">Email (Required)</label>
            <input type="email" id="customerEmail" class="form-control" placeholder="Enter your email" required>
            <small class="text-muted">We will send your ticket details to this email.</small>
          </div>

          <?php if(!$isOpen || $availableSeats <= 0): ?>
            <button class="register-btn mt-3" disabled>Sold Out</button>
          <?php else: ?>
            <button class="register-btn mt-3" id="payNowBtn">Pay & Register</button>
          <?php endif; ?>

          <div id="paypalArea" class="mt-3" style="display:none;">
            <div id="paypal-button-container"></div>
            <div id="resultBox"></div>
          </div>

          <div class="text-muted small mt-2">
            Available Seats: <b><?= (int)$availableSeats ?></b>
          </div>

        </div>
      </div>

    </div>
  </div>

  <div class="content-card mb-5">
    <div class="row g-0">

      <div class="col-lg-8">
        <div class="left-desc">

          <h5 class="mb-3"><?= htmlspecialchars($event['title']) ?></h5>

          <p class="text-muted">
            <?= nl2br(htmlspecialchars($event['description'] ?? "")) ?>
          </p>

          <?php if(!empty($eventParagraphs)): ?>
            <div class="mt-3">
              <?php foreach($eventParagraphs as $p): ?>
                <p class="text-muted mb-3"><?= nl2br(htmlspecialchars($p['paragraph_text'])) ?></p>
              <?php endforeach; ?>
            </div>
          <?php endif; ?>

          <hr>

          <p class="mb-2">
            <i class="bi bi-calendar-event me-2 text-success"></i>
            <b><?= htmlspecialchars($eventDayName) ?>, <?= htmlspecialchars($eventDateTxt) ?></b>
          </p>

          <p class="mb-2">
            <i class="bi bi-clock me-2 text-success"></i>
            Event Time: <b><?= htmlspecialchars($eventTimeTxt) ?></b>
          </p>

          <p class="mb-2">
            <i class="bi bi-geo-alt me-2 text-success"></i>
            <?= htmlspecialchars($event['subtitle']) ?>
          </p>

          <hr>

          <p class="mb-2"><b>Tickets:</b></p>
          <ul>
            <?php foreach($tickets as $t): ?>
              <li><?= htmlspecialchars($t['name']) ?>: $<?= number_format((float)$t['price'], 2) ?></li>
            <?php endforeach; ?>
          </ul>

        </div>
      </div>

      <div class="col-lg-4">
        <div class="sidebar">

          <div class="block">
            <h6>Date & Time</h6>
            <div class="fw-bold"><?= htmlspecialchars($eventDayName) ?>, <?= htmlspecialchars($eventDateTxt) ?></div>
            <div class="text-muted small"><?= htmlspecialchars($eventTimeTxt) ?> (Canada/Regina)</div>
          </div>

          <div class="block">
            <h6>Location</h6>
            <div class="fw-bold"><?= htmlspecialchars($event['subtitle']) ?></div>
          </div>

          <div class="block">
            <h6>Organizer</h6>
            <div class="fw-bold">IAOS Regina</div>
            <div class="text-muted small">info@iaosregina.com</div>
          </div>

        </div>
      </div>

    </div>
  </div>

</div>

<?php include "prayer_modal.php"; ?>
<?php include 'footer.php'; ?>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>

<!-- ✅ PayPal SDK -->
<script src="https://www.paypal.com/sdk/js?client-id=<?= htmlspecialchars($PAYPAL_CLIENT_ID) ?>&currency=<?= htmlspecialchars($PAY_CURRENCY) ?>"></script>

<script>
const AVAILABLE_SEATS = <?= (int)$availableSeats ?>;
const EVENT_ID        = <?= (int)$eventID ?>;
const resultBox       = document.getElementById("resultBox");

const ticketPrices = <?= json_encode(array_column($tickets, 'price', 'key')) ?>;

let qty = {};
Object.keys(ticketPrices).forEach(k => qty[k] = 0);

function calcTotal(){
  let t = 0;
  for(const k in qty){
    t += (qty[k] * parseFloat(ticketPrices[k]));
  }
  return t;
}

function totalTicketsCount(){
  let c = 0;
  for(const k in qty){
    c += qty[k];
  }
  return c;
}

function updateUI(){
  Object.keys(qty).forEach(k=>{
    const el = document.getElementById("qty_"+k);
    if(el) el.value = qty[k];
  });

  document.getElementById("totalAmount").innerText = calcTotal().toFixed(2);
}

document.querySelectorAll(".plus").forEach(btn=>{
  btn.addEventListener("click", ()=>{
    const key = btn.dataset.key;

    if(totalTicketsCount() >= AVAILABLE_SEATS){
      alert("No more seats available!");
      return;
    }

    qty[key] += 1;
    updateUI();
  });
});

document.querySelectorAll(".minus").forEach(btn=>{
  btn.addEventListener("click", ()=>{
    const key = btn.dataset.key;
    if(qty[key] > 0) qty[key] -= 1;
    updateUI();
  });
});

updateUI();

/* ==================================================
   ✅ PAYPAL RENDER FIX (Render only after click)
================================================== */
let paypalRendered = false;

function renderPaypal(customerEmail){
  if(paypalRendered) return;
  paypalRendered = true;

  paypal.Buttons({

    createOrder: function(data, actions) {
      const total = calcTotal().toFixed(2);

      if(parseFloat(total) <= 0){
        alert("Please select at least 1 ticket.");
        throw new Error("Total is zero");
      }

      return actions.order.create({
        purchase_units: [{
          amount: { value: total },
          description: "Event Tickets - EventID " + EVENT_ID
        }]
      });
    },

    onApprove: function(data, actions) {

      const total = calcTotal().toFixed(2);

      return fetch("selectedevent.php?action=capture", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          orderID: data.orderID,
          eventID: EVENT_ID,
          total: parseFloat(total),
          qty: qty,
          customer_email: customerEmail
        })
      })
      .then(res => res.json())
      .then(resp => {

        resultBox.style.display = "block";

        if(resp.status === "success"){
          resultBox.className = "successBox";
          resultBox.innerHTML = `
            <b>✅ Payment Successful!</b><br>
            <b>Order ID:</b> ${resp.orderID}<br>
            <b>Name:</b> ${resp.payerName}<br>
            <b>Email Sent To:</b> ${resp.customerEmail}<br>
            <b>Paid:</b> ${resp.currency} ${resp.amount}<br>
            <b>Email Sent:</b> ${resp.email_sent}<br>
            <br>
            <b>🎟️ Tickets Reserved:</b> ${totalTicketsCount()}
          `;
        } else {
          resultBox.className = "errorBox";
          resultBox.innerHTML = `
            <b>❌ Payment Failed</b><br>
            <b>Message:</b> ${resp.message}<br>
            <pre style="white-space:pre-wrap;">${JSON.stringify(resp.debug || resp, null, 2)}</pre>
          `;
        }

      })
      .catch(err => {
        resultBox.style.display = "block";
        resultBox.className = "errorBox";
        resultBox.innerHTML = "❌ Server Error: " + err;
        console.error(err);
      });

    },

    onCancel: function () {
      resultBox.style.display = "block";
      resultBox.className = "errorBox";
      resultBox.innerHTML = "⚠️ Payment cancelled by user.";
    },

    onError: function (err) {
      resultBox.style.display = "block";
      resultBox.className = "errorBox";
      resultBox.innerHTML = "❌ PayPal Error: " + err;
      console.error(err);
    }

  }).render("#paypal-button-container");
}

document.getElementById("payNowBtn")?.addEventListener("click", () => {
  const total = calcTotal();
  const emailField = document.getElementById("customerEmail");
  const customerEmail = (emailField.value || "").trim();

  if(total <= 0){
    alert("Please select at least 1 ticket.");
    return;
  }

  // ✅ validate email
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  if(!emailRegex.test(customerEmail)){
    alert("Please enter a valid email address.");
    emailField.focus();
    return;
  }

  document.getElementById("paypalArea").style.display = "block";
  document.getElementById("payNowBtn").style.display = "none";

  // ✅ NOW render PayPal button
  renderPaypal(customerEmail);
});
</script>

</body>
</html>
