<?php 
ob_start();
include "layout.php"; 
require_once "db.php"; 

if (!in_array($role, ['admin', 'eventController'])) {
    die("Access denied.");
}

$pdo = get_db();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

$message = "";

// ---------- ADD EVENT ----------
if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['title'])) {
    try {

        $title      = trim($_POST['title']);
        $subtitle   = trim($_POST['subtitle']);
        $eventDate  = $_POST['event_date'];
        $expiryDate = $_POST['expiry_date'];
        $capacity   = (int)$_POST['capacity'];
        $basePrice  = (float)$_POST['base_price'];
        $discount   = (float)$_POST['discount'];

        // ✅ NEW FIELDS
        $link        = trim($_POST['link'] ?? "");
        $description = trim($_POST['description'] ?? "");

        // ✅ TICKET OPTIONS (up to 5)
        $ticketHeadings = $_POST['ticket_headings'] ?? [];
        $ticketPrices   = $_POST['ticket_prices'] ?? [];

        // Prepare fixed 5 columns
        $tH = array_fill(1, 5, null);
        $tP = array_fill(1, 5, null);

        $index = 1;
        if (is_array($ticketHeadings) && is_array($ticketPrices)) {
            for ($i = 0; $i < count($ticketHeadings); $i++) {
                $h = trim($ticketHeadings[$i] ?? '');
                $p = trim($ticketPrices[$i] ?? '');

                if ($h !== "" && $p !== "" && is_numeric($p)) {
                    if ($index <= 5) {
                        $tH[$index] = $h;
                        $tP[$index] = number_format((float)$p, 2, '.', '');
                        $index++;
                    }
                }
            }
        }

        // Calculate member price (discounted)
        $memberPrice = $basePrice;
        if ($discount > 0) {
            $memberPrice = $basePrice - (($basePrice * $discount) / 100);
        }

        // Handle image upload
        $image = "";
        if (!empty($_FILES['image']['name'])) {
            $uploadDir = "uploads/events/";
            if (!is_dir($uploadDir)) mkdir($uploadDir, 0777, true);
            $filename = time() . "_" . basename($_FILES['image']['name']);
            $image = $uploadDir . $filename;
            move_uploaded_file($_FILES['image']['tmp_name'], $image);
        }

        /* ✅ TRANSACTION START */
        $pdo->beginTransaction();

        // ✅ INSERT EVENT (WITH TICKET OPTIONS)
        $stmt = $pdo->prepare("
            INSERT INTO events 
            (
              title, subtitle, event_date, expiry_date, image, capacity, 
              base_price, member_price, member_discount, link, description,

              ticketoption1heading, ticketoption1price,
              ticketoption2heading, ticketoption2price,
              ticketoption3heading, ticketoption3price,
              ticketoption4heading, ticketoption4price,
              ticketoption5heading, ticketoption5price
            )
            VALUES 
            (
              ?, ?, ?, ?, ?, ?, 
              ?, ?, ?, ?, ?,

              ?, ?,
              ?, ?,
              ?, ?,
              ?, ?,
              ?, ?
            )
        ");

        $stmt->execute([
            $title, $subtitle, $eventDate, $expiryDate, $image, 
            $capacity, $basePrice, $memberPrice, $discount,
            $link, $description,

            $tH[1], $tP[1],
            $tH[2], $tP[2],
            $tH[3], $tP[3],
            $tH[4], $tP[4],
            $tH[5], $tP[5],
        ]);

        $eventId = $pdo->lastInsertId();

        /* ✅ INSERT PARAGRAPHS (FIXED) */
        if (isset($_POST['paragraphs']) && is_array($_POST['paragraphs'])) {

            $paraStmt = $pdo->prepare("
                INSERT INTO event_paragraphs (id, event_id, paragraph_text) 
                VALUES (NULL, ?, ?)
            ");

            foreach ($_POST['paragraphs'] as $para) {
                $para = trim($para);
                if ($para !== "") {
                    $paraStmt->execute([$eventId, $para]);
                }
            }
        }

        /* ✅ COMMIT */
        $pdo->commit();

        $message = "✅ Event created successfully!";

    } catch (Exception $e) {

        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }

        $message = "❌ Error: " . $e->getMessage();
    }
}

// ---------- FETCH EVENTS ----------
$events = $pdo->query("SELECT * FROM events ORDER BY id DESC")->fetchAll();

// ---------- AUTO UPDATE STATUS ----------
foreach ($events as $e) {
    if (strtotime($e['expiry_date']) < time() || $e['booked'] >= $e['capacity']) {
        $pdo->prepare("UPDATE events SET status='soldout' WHERE id=?")->execute([$e['id']]);
    }
}
?>

<div class="container-fluid" style="margin-top:50px;">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h3><i class="bi bi-calendar-event text-white"></i> Event Controller</h3>
  </div>

  <?php if($message): ?>
    <div class="alert alert-success"><?= htmlspecialchars($message) ?></div>
  <?php endif; ?>

  <!-- 🟢 Add New Event -->
  <div class="card p-4 mb-5 shadow-lg" style="background:rgba(255,255,255,0.1);border:none;">
    <h5 class="mb-3 text-white"><i class="bi bi-plus-circle text-white"></i> Add New Event (Prices in <b>CAD $</b>)</h5>
    <form method="POST" enctype="multipart/form-data">

      <!-- Title + Subtitle -->
      <div class="row mb-3">
        <div class="col-md-6">
          <label class="text-white">Event Title</label>
          <input type="text" name="title" class="form-control" required>
        </div>
        <div class="col-md-6">
          <label class="text-white">Event Location</label>
          <input type="text" name="subtitle" class="form-control">
        </div>
      </div>

      <!-- DATES / PRICES -->
      <div class="row mb-3">
        <div class="col-md-3">
          <label class="text-white">Event Date</label>
          <input type="datetime-local" name="event_date" class="form-control" required>
        </div>
        <div class="col-md-3">
          <label class="text-white">Ticket Expiry</label>
          <input type="datetime-local" name="expiry_date" class="form-control" required>
        </div>
        <div class="col-md-2">
          <label class="text-white">Capacity</label>
          <input type="number" name="capacity" class="form-control" min="1" required>
        </div>
        <div class="col-md-2">
          <label class="text-white">Base Price (CAD)</label>
          <input type="number" step="0.01" name="base_price" class="form-control" required>
        </div>
        <div class="col-md-2">
          <label class="text-white">Member Discount (%)</label>
          <input type="number" step="0.01" name="discount" class="form-control">
        </div>
      </div>

      <!-- Member Price + Image -->
      <div class="row mb-3">
        <div class="col-md-3">
          <label class="text-white">Member Price (Auto)</label>
          <input type="text" id="memberPrice" class="form-control bg-light" readonly>
        </div>
        <div class="col-md-3">
          <label class="text-white">Upload Image</label>
          <input type="file" name="image" accept="image/*" class="form-control">
        </div>
      </div>

      <!-- ✅ NEW LINK + DESCRIPTION -->
      <div class="row mb-3">
        <div class="col-md-6">
          <label class="text-white">Event Link (Optional)</label>
          <input type="url" name="link" class="form-control" placeholder="https://example.com">
        </div>

        <div class="col-md-6">
          <label class="text-white">Short Description</label>
          <input type="text" name="description" class="form-control" placeholder="Short event description...">
        </div>
      </div>

      <!-- ✅✅✅ TICKET OPTIONS (LIKE PARAGRAPHS) -->
      <div class="mb-3">
        <label class="text-white fw-bold mb-2 d-block">Ticket Options (Heading + Price)</label>

        <div id="ticketOptionsContainer">
          <div class="row g-2 mb-2 ticketRow">
            <div class="col-md-7">
              <input type="text" name="ticket_headings[]" class="form-control" placeholder="Ticket Name (e.g. Adult Admission)">
            </div>
            <div class="col-md-3">
              <input type="number" step="0.01" name="ticket_prices[]" class="form-control" placeholder="Price CAD">
            </div>
            <div class="col-md-2">
              <button type="button" class="btn btn-danger w-100 removeTicketBtn" disabled>
                <i class="bi bi-trash"></i>
              </button>
            </div>
          </div>
        </div>

        <button type="button" class="btn btn-outline-light btn-sm" id="addTicketBtn">
          <i class="bi bi-plus-circle text-white"></i> Add Another Ticket Option
        </button>

        <small class="text-white d-block mt-1" style="opacity:0.8;">
          Max 5 ticket options allowed.
        </small>
      </div>

      <!-- Paragraph Section -->
      <div id="paragraphsContainer" class="mb-3">
        <label class="text-white">Paragraphs</label>
        <textarea name="paragraphs[]" class="form-control mb-2" rows="2" placeholder="Write paragraph..."></textarea>
      </div>

      <button type="button" class="btn btn-outline-light btn-sm" id="addParagraphBtn">
        <i class="bi bi-plus-circle text-white"></i> Add Another Paragraph
      </button>

      <div class="text-end mt-3">
        <button class="btn btn-success px-4"><i class="bi bi-save"></i> Save Event</button>
      </div>

    </form>
  </div>

  <!-- 📋 Event List -->
  <div class="card p-4 shadow-lg" style="background:rgba(255,255,255,0.1);border:none;">
    <h5 class="mb-3 text-white"><i class="bi bi-list-ul text-white"></i> Existing Events</h5>
    <div class="table-responsive">

      <table class="table table-striped table-hover align-middle text-center bg-light text-dark">
        <thead class="table-success text-dark">
          <tr>
            <th>ID</th>
            <th>Image</th>
            <th>Title</th>
            <th>Event Date</th>
            <th>Expiry</th>
            <th>Capacity</th>
            <th>Booked</th>
            <th>Non-Member Price</th>
            <th>Member Price</th>
            <th>Discount</th>

            <th>Link</th>
            <th>Description</th>

            <th>Status</th>
            <th>Actions</th>
          </tr>
        </thead>

        <tbody>
          <?php foreach($events as $e): ?>
          <tr>
            <td><?= $e['id'] ?></td>

            <td>
              <?php if($e['image']): ?>
                <img src="<?= $e['image'] ?>" width="80" height="60" style="object-fit:cover;border-radius:6px;">
              <?php else: ?>
                <span class="text-muted">No Image</span>
              <?php endif; ?>
            </td>

            <td><?= htmlspecialchars($e['title']) ?></td>
            <td><?= date("Y-m-d", strtotime($e['event_date'])) ?></td>

            <td>
              <?php 
                $expiry = strtotime($e['expiry_date']);
                $now = time();
                if ($now > $expiry) {
                  echo "<span class='badge bg-secondary'>Closed</span>";
                } else {
                  echo date("Y-m-d H:i", $expiry);
                }
              ?>
            </td>

            <td><?= $e['capacity'] ?></td>
            <td><?= $e['booked'] ?></td>
            <td><b>$<?= number_format($e['base_price'], 2) ?></b></td>
            <td><b>$<?= number_format($e['member_price'], 2) ?></b></td>
            <td><?= $e['member_discount'] ?>%</td>

            <td>
              <?php if(!empty($e['link'])): ?>
                <a href="<?= htmlspecialchars($e['link']) ?>" target="_blank" class="btn btn-sm btn-primary">
                  <i class="bi bi-link-45deg"></i> Open
                </a>
              <?php else: ?>
                <span class="text-muted">No Link</span>
              <?php endif; ?>
            </td>

            <td><?= htmlspecialchars($e['description']) ?: "<span class='text-muted'>No Description</span>" ?></td>

            <td>
              <?php if($e['status'] == 'soldout'): ?>
                <span class="badge bg-danger">Sold Out</span>
              <?php else: ?>
                <span class="badge bg-success">Open</span>
              <?php endif; ?>
            </td>

            <td>
                <a href="edit_event.php?id=<?= $e['id'] ?>" class="btn btn-sm btn-warning">
                    <i class="bi bi-pencil-square"></i>
                </a>
            </td>
          </tr>
          <?php endforeach; ?>
        </tbody>

      </table>

    </div>
  </div>
</div>

<script>
// ✅ Add Paragraph
document.getElementById("addParagraphBtn").addEventListener("click", function() {
  const container = document.getElementById("paragraphsContainer");
  const textarea = document.createElement("textarea");
  textarea.name = "paragraphs[]";
  textarea.className = "form-control mb-2";
  textarea.rows = 2;
  textarea.placeholder = "Write another paragraph...";
  container.appendChild(textarea);
});

// ✅ Ticket Options Add/Remove (MAX 5)
document.getElementById("addTicketBtn").addEventListener("click", function() {

  const container = document.getElementById("ticketOptionsContainer");
  const existing = container.querySelectorAll(".ticketRow").length;

  if(existing >= 5){
    alert("Maximum 5 ticket options allowed!");
    return;
  }

  const row = document.createElement("div");
  row.className = "row g-2 mb-2 ticketRow";
  row.innerHTML = `
    <div class="col-md-7">
      <input type="text" name="ticket_headings[]" class="form-control" placeholder="Ticket Name (e.g. Kids Admission)">
    </div>
    <div class="col-md-3">
      <input type="number" step="0.01" name="ticket_prices[]" class="form-control" placeholder="Price CAD">
    </div>
    <div class="col-md-2">
      <button type="button" class="btn btn-danger w-100 removeTicketBtn">
        <i class="bi bi-trash"></i>
      </button>
    </div>
  `;
  container.appendChild(row);

  updateRemoveButtons();
});

function updateRemoveButtons(){
  const rows = document.querySelectorAll(".ticketRow");
  rows.forEach((r, idx) => {
    const btn = r.querySelector(".removeTicketBtn");
    if(!btn) return;

    btn.disabled = (rows.length === 1);
    btn.onclick = () => {
      r.remove();
      updateRemoveButtons();
    };
  });
}
updateRemoveButtons();

// ✅ Auto calculate member price
document.querySelector('input[name="base_price"]').addEventListener('input', updateMemberPrice);
document.querySelector('input[name="discount"]').addEventListener('input', updateMemberPrice);

function updateMemberPrice() {
  const base = parseFloat(document.querySelector('input[name="base_price"]').value) || 0;
  const disc = parseFloat(document.querySelector('input[name="discount"]').value) || 0;
  const memberPrice = (base - (base * disc / 100)).toFixed(2);
  document.getElementById('memberPrice').value = memberPrice > 0 
      ? "$" + memberPrice + " CAD" 
      : "$0.00 CAD";
}
</script>

</div></body></html>
