<?php
/************************************************************
 * newsletter_upload_image.php
 * ✅ Upload image for Quill editor (FREE)
 * ✅ Returns JSON {status, url}
 * ✅ Secure: file type check + extension check + size limit
 * ✅ Saves into: /images/newsletter_uploads/
 ************************************************************/

session_start();
require_once "db.php"; // not required but keep for consistency

header("Content-Type: application/json");

// ✅ (Optional) protect admin only
// if(empty($_SESSION['admin_logged_in'])) {
//   echo json_encode(["status"=>"error","message"=>"Unauthorized"]);
//   exit;
// }

if($_SERVER["REQUEST_METHOD"] !== "POST"){
  echo json_encode(["status"=>"error","message"=>"Invalid request"]);
  exit;
}

if(!isset($_FILES["image"]) || empty($_FILES["image"]["name"])){
  echo json_encode(["status"=>"error","message"=>"No file uploaded"]);
  exit;
}

$file = $_FILES["image"];

// ✅ Max upload: 10MB
$maxSize = 10 * 1024 * 1024;
if($file["size"] > $maxSize){
  echo json_encode(["status"=>"error","message"=>"Image too large (max 10MB)"]);
  exit;
}

// ✅ Allowed extensions
$allowedExt = ["jpg","jpeg","png","webp","gif"];
$ext = strtolower(pathinfo($file["name"], PATHINFO_EXTENSION));

if(!in_array($ext, $allowedExt)){
  echo json_encode(["status"=>"error","message"=>"Invalid file type"]);
  exit;
}

// ✅ Validate real image
$tmpPath = $file["tmp_name"];
$imgInfo = @getimagesize($tmpPath);
if($imgInfo === false){
  echo json_encode(["status"=>"error","message"=>"File is not a valid image"]);
  exit;
}

// ✅ Upload folder
$uploadDir = __DIR__ . "/../images/newsletter_uploads/";
if(!is_dir($uploadDir)){
  mkdir($uploadDir, 0777, true);
}

// ✅ Create unique file name
$newName = "nl_" . date("Ymd_His") . "_" . bin2hex(random_bytes(6)) . "." . $ext;
$savePath = $uploadDir . $newName;

// ✅ Move uploaded file
if(!move_uploaded_file($tmpPath, $savePath)){
  echo json_encode(["status"=>"error","message"=>"Upload failed"]);
  exit;
}

/* ✅ OPTIONAL: Keep HD but reduce huge images width (safe for email)
   - You can ENABLE this block if you want to resize big images
   - It keeps quality nice for newsletter emails
*/
$resizeEnabled = false; // ✅ set TRUE if you want resize
$maxWidth = 1600;       // ✅ HD email width limit

if($resizeEnabled){
  try{

    $w = $imgInfo[0];
    $h = $imgInfo[1];

    if($w > $maxWidth){
      $newW = $maxWidth;
      $newH = (int)(($h / $w) * $newW);

      // load image based on ext
      if($ext === "jpg" || $ext === "jpeg"){
        $src = imagecreatefromjpeg($savePath);
      } elseif($ext === "png"){
        $src = imagecreatefrompng($savePath);
      } elseif($ext === "webp"){
        $src = imagecreatefromwebp($savePath);
      } elseif($ext === "gif"){
        $src = imagecreatefromgif($savePath);
      } else {
        $src = null;
      }

      if($src){
        $dst = imagecreatetruecolor($newW, $newH);

        // keep png transparency
        if($ext === "png"){
          imagealphablending($dst, false);
          imagesavealpha($dst, true);
        }

        imagecopyresampled($dst, $src, 0,0,0,0, $newW,$newH, $w,$h);

        // save back
        if($ext === "jpg" || $ext === "jpeg"){
          imagejpeg($dst, $savePath, 90); // ✅ high quality
        } elseif($ext === "png"){
          imagepng($dst, $savePath, 6);
        } elseif($ext === "webp"){
          imagewebp($dst, $savePath, 90);
        } elseif($ext === "gif"){
          imagegif($dst, $savePath);
        }

        imagedestroy($src);
        imagedestroy($dst);
      }
    }

  } catch(Exception $e){
    // ignore resize errors
  }
}

// ✅ Public URL for frontend
// Your admin is inside: /admin_pannel/
// Image folder is: /images/newsletter_uploads/
$imageUrl = "../images/newsletter_uploads/" . $newName;

echo json_encode([
  "status" => "success",
  "url"    => $imageUrl,
  "name"   => $newName
]);
exit;
?>
